//! The business logic of the CI broker.
//!
//! This is type and module of its own to facilitate automated
//! testing.

use std::{collections::HashMap, path::Path};

use log::debug;
use time::{macros::format_description, OffsetDateTime};

use radicle::prelude::RepoId;

use crate::{
    adapter::Adapter,
    db::Db,
    error::BrokerError,
    msg::{PatchEvent, PushEvent, Request},
    pages::StatusPage,
    run::{Run, Whence},
};

/// A CI broker.
///
/// The broker gets repository change events from the local Radicle
/// node, and executes the appropriate adapter to run CI on the
/// repository.
#[derive(Debug)]
pub struct Broker {
    default_adapter: Option<Adapter>,
    adapters: HashMap<RepoId, Adapter>,
    db: Db,
}

impl Broker {
    #[allow(clippy::result_large_err)]
    pub fn new(db_filename: &Path) -> Result<Self, BrokerError> {
        debug!("broker database in {}", db_filename.display());
        Ok(Self {
            default_adapter: None,
            adapters: HashMap::new(),
            db: Db::new(db_filename)?,
        })
    }

    #[allow(clippy::result_large_err)]
    pub fn all_runs(&mut self) -> Result<Vec<Run>, BrokerError> {
        self.db.all_runs().map_err(BrokerError::Db)
    }

    pub fn set_default_adapter(&mut self, adapter: &Adapter) {
        self.default_adapter = Some(adapter.clone());
    }

    pub fn default_adapter(&self) -> Option<&Adapter> {
        self.default_adapter.as_ref()
    }

    pub fn set_repository_adapter(&mut self, rid: &RepoId, adapter: &Adapter) {
        self.adapters.insert(*rid, adapter.clone());
    }

    pub fn adapter(&self, rid: &RepoId) -> Option<&Adapter> {
        self.adapters.get(rid).or(self.default_adapter.as_ref())
    }

    #[allow(clippy::result_large_err)]
    pub fn execute_ci(
        &mut self,
        trigger: &Request,
        status: &mut StatusPage,
    ) -> Result<Run, BrokerError> {
        let run = match trigger {
            Request::Trigger {
                common,
                push,
                patch,
            } => {
                let rid = &common.repository.id;
                if let Some(adapter) = self.adapter(rid) {
                    let whence = if let Some(PushEvent {
                        pusher: _,
                        before: _,
                        after,
                        branch: _,
                        commits: _,
                    }) = push
                    {
                        Whence::branch("push-event-has-no-branch-name", *after)
                    } else if let Some(PatchEvent { action: _, patch }) = patch {
                        Whence::patch(patch.id, patch.after)
                    } else {
                        panic!("neither push not patch event");
                    };

                    let mut run = Run::new(*rid, &common.repository.name, whence, now());
                    adapter.run(trigger, &mut run, status)?;
                    run
                } else {
                    return Err(BrokerError::NoAdapter(*rid));
                }
            }
        };
        self.db.push_run(&run)?;

        Ok(run)
    }
}

fn now() -> String {
    let fmt = format_description!("[year]-[month]-[day] [hour]:[minute]:[second]Z");
    OffsetDateTime::now_utc().format(fmt).expect("format time")
}

#[cfg(test)]
mod test {
    use std::path::Path;
    use tempfile::tempdir;

    use super::{Adapter, Broker, RepoId};
    use crate::{
        msg::{RunId, RunResult},
        pages::{PageBuilder, StatusPage},
        run::RunState,
        test::{mock_adapter, trigger_request, TestResult},
    };

    fn broker(filename: &Path) -> Broker {
        Broker::new(filename).unwrap()
    }

    fn rid() -> RepoId {
        const RID: &str = "rad:zwTxygwuz5LDGBq255RA2CbNGrz8";
        RepoId::from_urn(RID).unwrap()
    }

    fn rid2() -> RepoId {
        const RID: &str = "rad:z3gqcJUoA1n9HaHKufZs5FCSGazv5";
        RepoId::from_urn(RID).unwrap()
    }

    fn status_page() -> StatusPage {
        PageBuilder::default()
            .node_alias("test.alias")
            .build()
            .unwrap()
    }

    #[test]
    fn has_no_adapters_initially() -> TestResult<()> {
        let tmp = tempdir().unwrap();
        let db = tmp.path().join("db.db");
        let broker = broker(&db);
        let rid = rid();
        assert_eq!(broker.adapter(&rid), None);
        Ok(())
    }

    #[test]
    fn adds_adapter() -> TestResult<()> {
        let tmp = tempdir().unwrap();
        let db = tmp.path().join("db.db");
        let mut broker = broker(&db);

        let adapter = Adapter::default();
        let rid = rid();
        broker.set_repository_adapter(&rid, &adapter);
        assert_eq!(broker.adapter(&rid), Some(&adapter));
        Ok(())
    }

    #[test]
    fn does_not_find_unknown_repo() -> TestResult<()> {
        let tmp = tempdir().unwrap();
        let db = tmp.path().join("db.db");
        let mut broker = broker(&db);

        let adapter = Adapter::default();
        let rid = rid();
        let rid2 = rid2();
        broker.set_repository_adapter(&rid, &adapter);
        assert_eq!(broker.adapter(&rid2), None);
        Ok(())
    }

    #[test]
    fn does_not_have_a_default_adapter_initially() -> TestResult<()> {
        let tmp = tempdir().unwrap();
        let db = tmp.path().join("db.db");
        let broker = broker(&db);

        assert_eq!(broker.default_adapter(), None);
        Ok(())
    }

    #[test]
    fn sets_a_default_adapter_initially() -> TestResult<()> {
        let tmp = tempdir().unwrap();
        let db = tmp.path().join("db.db");
        let mut broker = broker(&db);

        let adapter = Adapter::default();
        broker.set_default_adapter(&adapter);
        assert_eq!(broker.default_adapter(), Some(&adapter));
        Ok(())
    }

    #[test]
    fn finds_default_adapter_for_unknown_repo() -> TestResult<()> {
        let tmp = tempdir().unwrap();
        let db = tmp.path().join("db.db");
        let mut broker = broker(&db);

        let adapter = Adapter::default();
        broker.set_default_adapter(&adapter);

        let rid = rid();
        assert_eq!(broker.adapter(&rid), Some(&adapter));
        Ok(())
    }

    #[test]
    fn executes_adapter() -> TestResult<()> {
        const ADAPTER: &str = r#"#!/bin/bash
echo '{"response":"triggered","run_id":{"id":"xyzzy"}}'
echo '{"response":"finished","result":"success"}'
"#;

        let tmp = tempdir()?;
        let bin = tmp.path().join("adapter.sh");
        let adapter = mock_adapter(&bin, ADAPTER)?;

        let tmp = tempdir().unwrap();
        let db = tmp.path().join("db.db");
        let mut broker = broker(&db);
        broker.set_default_adapter(&adapter);

        let trigger = trigger_request()?;

        let mut status = status_page();
        let x = broker.execute_ci(&trigger, &mut status);
        assert!(x.is_ok());
        let run = x.unwrap();
        assert_eq!(run.adapter_run_id(), Some(&RunId::from("xyzzy")));
        assert_eq!(run.state(), RunState::Finished);
        assert_eq!(run.result(), Some(&RunResult::Success));

        Ok(())
    }
}
