//! Persistent database for CI run information.

use std::{
    fmt,
    path::{Path, PathBuf},
};

use sqlite::{Connection, State};

use crate::run::Run;

const CREATE_TABLES: &str =
    "CREATE TABLE IF NOT EXISTS ci_runs (run_id TEXT PRIMARY KEY, json TEXT)";

const INSERT_ROW: &str = "INSERT OR REPLACE INTO ci_runs (run_id, json) VALUES (:id, :json)";

const ALL_RUNS: &str = "SELECT json FROM ci_runs";

pub struct Db {
    filename: PathBuf,
    conn: Connection,
}

impl fmt::Debug for Db {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> Result<(), fmt::Error> {
        write!(f, "<Db:{}>", self.filename.display())
    }
}

impl Db {
    pub fn new(filename: &Path) -> Result<Self, DbError> {
        eprintln!("open {}", filename.display());
        let conn = sqlite::open(filename).map_err(|e| DbError::open(filename, e))?;

        eprintln!("create tables");
        conn.execute(CREATE_TABLES)
            .map_err(|e| DbError::create_tables(CREATE_TABLES, e))?;

        Ok(Db {
            conn,
            filename: filename.into(),
        })
    }

    pub fn push_run(&mut self, run: &Run) -> Result<(), DbError> {
        let json = serde_json::to_string(&run).map_err(DbError::to_json)?;

        let mut stmt = self
            .conn
            .prepare(INSERT_ROW)
            .map_err(|e| DbError::prepare(INSERT_ROW, e))?;

        let run_id = format!("{}", run.adapter_run_id().unwrap());
        stmt.bind((":id", run_id.as_str()))
            .map_err(|e| DbError::bind(":id", e))?;
        stmt.bind((":json", json.as_str()))
            .map_err(|e| DbError::bind(":json", e))?;
        stmt.next().map_err(DbError::insert_run)?;

        Ok(())
    }

    pub fn all_runs(&mut self) -> Result<Vec<Run>, DbError> {
        let mut stmt = self
            .conn
            .prepare(ALL_RUNS)
            .map_err(|e| DbError::prepare(ALL_RUNS, e))?;

        let mut runs = vec![];
        while let Ok(State::Row) = stmt.next() {
            let json: String = stmt.read("json").map_err(DbError::get_run)?;
            let run: Run = serde_json::from_str(&json).map_err(DbError::from_json)?;
            runs.push(run);
        }

        Ok(runs)
    }
}

/// All errors from this module.
#[derive(Debug, thiserror::Error)]
pub enum DbError {
    /// Error opening a database file.
    #[error("failed to open SQLite database {0}")]
    Open(PathBuf, #[source] sqlite::Error),

    /// Error creating tables.
    #[error("failed to create tables: {0}")]
    CreateTables(&'static str, #[source] sqlite::Error),

    /// Error preparing an SQL statement.
    #[error("failed to prepare SQL statement {0}")]
    Prepare(&'static str, #[source] sqlite::Error),

    /// Error binding a value to an SQL statement placeholder.
    #[error("failed to bind a value to SQL statement placeholder {0}")]
    Bind(&'static str, #[source] sqlite::Error),

    /// Error inserting or updating a run in SQL database.
    #[error("failed to insert or update a run in SQL database")]
    InsertRun(#[source] sqlite::Error),

    /// Error getting a run from SQL query.
    #[error("failed to get CI run from SQL query result")]
    GetRun(#[source] sqlite::Error),

    /// Error serializing a [`Run`]` into a string.
    #[error("failed to serialize a CI run into JSON")]
    ToJson(#[source] serde_json::Error),

    /// Error deserializing a [`Run`]` from a string.
    #[error("failed to parse JSON as a CI run")]
    FromJson(#[source] serde_json::Error),
}

impl DbError {
    fn open(filename: &Path, e: sqlite::Error) -> Self {
        Self::Open(filename.into(), e)
    }

    fn create_tables(query: &'static str, e: sqlite::Error) -> Self {
        Self::CreateTables(query, e)
    }

    fn prepare(stmt: &'static str, e: sqlite::Error) -> Self {
        Self::Prepare(stmt, e)
    }

    fn bind(placeholder: &'static str, e: sqlite::Error) -> Self {
        Self::Bind(placeholder, e)
    }

    fn insert_run(e: sqlite::Error) -> Self {
        Self::InsertRun(e)
    }

    fn to_json(e: serde_json::Error) -> Self {
        Self::ToJson(e)
    }

    fn from_json(e: serde_json::Error) -> Self {
        Self::FromJson(e)
    }

    fn get_run(e: sqlite::Error) -> Self {
        Self::GetRun(e)
    }
}
