//! Possible errors returned by the CI broker library.
//!
//! Each module has its own error type, this module collects them
//! together into a unified type, for callers who don't care to handle
//! module errors in specific ways.

use std::path::PathBuf;

use radicle::prelude::RepoId;

use crate::{
    adapter::AdapterError,
    config::ConfigError,
    db::DbError,
    msg::{MessageError, Request},
    pages::PageError,
};

/// All possible errors from the CI broker messages.
#[derive(Debug, thiserror::Error)]
#[allow(clippy::large_enum_variant)]
pub enum BrokerError {
    /// A configuration related error.
    #[error(transparent)]
    Config(#[from] ConfigError),

    /// A message related error.
    #[error(transparent)]
    Message(#[from] MessageError),

    /// An adapter related error.
    #[error(transparent)]
    Adapter(#[from] AdapterError),

    /// Error from an node event subscriber.
    #[error(transparent)]
    NodeEvent(#[from] crate::event::NodeEventError),

    /// Error from Radicle.
    #[error(transparent)]
    RadicleProfile(#[from] radicle::profile::Error),

    /// Error from spawning a sub-process.
    #[error("failed to spawn a CI adapter sub-process: {0}")]
    SpawnAdapter(PathBuf, #[source] std::io::Error),

    /// Usage error.
    #[error("usage: radicle-ci-broker CONFIG")]
    Usage,

    /// Default adapter is not in list of adapters.
    #[error("default adapter is not in list of adapters")]
    UnknownDefaultAdapter(String),

    /// No adapter set for repository and no default adapter set.
    #[error("could not determine what adapter to use for repository {0}")]
    NoAdapter(RepoId),

    /// Request is not a trigger message.
    #[error("tried to execute CI based on a message that is not a trigger one: {0:#?}")]
    NotTrigger(Request),

    /// Could not convert repository ID from string.
    #[error("failed to understand repository id {0:?}")]
    BadRepoId(String, #[source] radicle::identity::IdError),

    /// Status page error.
    #[error(transparent)]
    StatusPage(#[from] PageError),

    /// Database error.
    #[error(transparent)]
    Db(#[from] DbError),
}
