use std::fmt;

use serde::{Deserialize, Serialize};

use radicle::git::Oid;
use radicle::prelude::RepoId;

use crate::msg::{RunId, RunResult};

#[derive(Debug, Clone, Eq, PartialEq, Serialize, Deserialize)]
pub struct Run {
    adapter_run_id: Option<RunId>,
    repo_id: RepoId,
    repo_alias: String,
    timestamp: String,
    whence: Whence,
    state: RunState,
    result: Option<RunResult>,
}

impl Run {
    /// Create a new run.
    pub fn new(repo_id: RepoId, alias: &str, whence: Whence, timestamp: String) -> Self {
        Self {
            adapter_run_id: None,
            repo_id,
            repo_alias: alias.into(),
            timestamp,
            whence,
            state: RunState::Triggered,
            result: None,
        }
    }

    /// Return the repo alias.
    pub fn repo_alias(&self) -> &str {
        &self.repo_alias
    }

    /// Return the repo id.
    pub fn repo_id(&self) -> RepoId {
        self.repo_id
    }

    /// Return timestamp of run.
    pub fn timestamp(&self) -> &str {
        &self.timestamp
    }

    /// Return where the commit came from.
    pub fn whence(&self) -> &Whence {
        &self.whence
    }

    /// Set the run id assigned by the adapter.
    pub fn set_adapter_run_id(&mut self, run_id: RunId) {
        self.adapter_run_id = Some(run_id);
    }

    /// Return the run id assigned by the adapter, if any.
    pub fn adapter_run_id(&self) -> Option<&RunId> {
        self.adapter_run_id.as_ref()
    }

    /// Return the state of the CI run.
    pub fn state(&self) -> RunState {
        self.state
    }

    /// Set the state of the run.
    pub fn set_state(&mut self, state: RunState) {
        self.state = state;
    }

    /// Set the result of the CI run.
    pub fn set_result(&mut self, result: RunResult) {
        self.result = Some(result);
    }

    /// Return the result of the CI run.
    pub fn result(&self) -> Option<&RunResult> {
        self.result.as_ref()
    }
}

/// State of CI run.
#[derive(Debug, Copy, Clone, Eq, PartialEq, Serialize, Deserialize)]
#[serde(rename_all = "snake_case")]
pub enum RunState {
    /// Run has been triggered, but has not yet been started.
    Triggered,

    /// Run is currently running.
    Running,

    /// Run has finished. It may or may not have succeeded.
    Finished,
}

impl fmt::Display for RunState {
    fn fmt(&self, f: &mut fmt::Formatter) -> Result<(), fmt::Error> {
        let s = match self {
            Self::Finished => "finished",
            Self::Running => "running",
            Self::Triggered => "triggered",
        };
        write!(f, "{}", s)
    }
}

/// Where did the commit come that CI is run for?
#[derive(Debug, Clone, Eq, PartialEq, Serialize, Deserialize)]
pub enum Whence {
    Branch { name: String, commit: Oid },
    Patch { patch: Oid, commit: Oid },
}

impl Whence {
    pub fn branch(name: &str, commit: Oid) -> Self {
        Self::Branch {
            name: name.into(),
            commit,
        }
    }

    pub fn patch(patch: Oid, commit: Oid) -> Self {
        Self::Patch { patch, commit }
    }
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn serialize_run_state() {
        let s = serde_json::to_string(&RunState::Finished).unwrap();
        assert_eq!(s, r#""finished""#);
    }
}
