//! Show broker events that are allowed by a filter. This is meant to
//! be helpful for testing a filter configuration.

use log::info;
use std::{error::Error, path::PathBuf};

use radicle::prelude::Profile;
use radicle_ci_broker::{config::Config, error::BrokerError, event::NodeEventSource};

fn main() {
    if let Err(e) = fallible_main() {
        eprintln!("ERROR: {}", e);
        let mut e = e.source();
        while let Some(source) = e {
            eprintln!("caused by: {}", source);
            e = source.source();
        }
    }
}

fn fallible_main() -> Result<(), BrokerError> {
    pretty_env_logger::init();

    let mut args = std::env::args().skip(1);
    let filename: PathBuf = if let Some(filename) = args.next() {
        PathBuf::from(filename)
    } else {
        return Err(BrokerError::Usage);
    };

    info!("using file: {:?}", filename);

    let config = Config::load(&filename)?;

    let profile = Profile::load()?;
    let mut source = NodeEventSource::new(&profile)?;
    for filter in config.filters.iter() {
        source.allow(filter.clone());
    }

    // This loop ends when there's an error, e.g., failure to read an
    // event from the node.
    loop {
        for e in source.event()? {
            println!("{:#?}", e);
        }
    }
}
