//! Messages for communicating with the adapter.
//!
//! The broker spawns an adapter child process, and sends it a request
//! via the child's stdin. The child sends responses via its stdout,
//! which the broker reads and processes. These messages are
//! represented using the types in this module.
//!
//! The types in this module are meant to be useful for anyone writing
//! a Radicle CI adapter.

#![deny(missing_docs)]

use std::{
    fmt,
    hash::{Hash, Hasher},
    io::{BufRead, BufReader, Read, Write},
};

use log::debug;
use serde::{Deserialize, Serialize};
use uuid::Uuid;

pub use radicle::{
    git::Oid,
    prelude::{NodeId, RepoId},
};
use radicle::{
    identity::Did,
    node::{Alias, AliasStore},
    patch::{self, RevisionId},
    storage::{git::paths, ReadRepository, ReadStorage},
    Profile,
};

use crate::event::{is_patch_update, push_branch, BrokerEvent};

// This gets put into every [`Request`] message so the adapter can
// detect its getting a message it knows how to handle.
const PROTOCOL_VERSION: usize = 1;

/// The type of a run identifier. For maximum generality, this is a
/// string rather than an integer.
///
/// # Example
/// ```rust
/// use radicle_ci_broker::msg::RunId;
/// let id = RunId::from("abracadabra");
/// println!("{}", id.to_string());
/// ```
#[derive(Debug, Clone, Eq, PartialEq, Serialize, Deserialize)]
pub struct RunId {
    id: String,
}

impl Default for RunId {
    fn default() -> Self {
        Self {
            id: Uuid::new_v4().to_string(),
        }
    }
}

impl Hash for RunId {
    fn hash<H: Hasher>(&self, h: &mut H) {
        self.id.hash(h);
    }
}

impl From<&str> for RunId {
    fn from(id: &str) -> Self {
        Self { id: id.into() }
    }
}

impl fmt::Display for RunId {
    fn fmt(&self, f: &mut fmt::Formatter) -> Result<(), fmt::Error> {
        write!(f, "{}", self.id)
    }
}

/// The result of a CI run.
#[derive(Debug, Clone, Eq, PartialEq, Serialize, Deserialize)]
#[serde(deny_unknown_fields)]
#[serde(rename_all = "snake_case")]
#[non_exhaustive]
pub enum RunResult {
    /// CI run was successful.
    Success,

    /// CI run failed.
    Failure,
}

impl fmt::Display for RunResult {
    fn fmt(&self, f: &mut fmt::Formatter) -> Result<(), fmt::Error> {
        match self {
            Self::Failure => write!(f, "failure"),
            Self::Success => write!(f, "success"),
        }
    }
}

/// Build a [`Request`].
#[derive(Debug, Default)]
pub struct RequestBuilder<'a> {
    profile: Option<&'a Profile>,
    event: Option<&'a BrokerEvent>,
}

impl<'a> RequestBuilder<'a> {
    /// Set the node profile to use.
    pub fn profile(mut self, profile: &'a Profile) -> Self {
        self.profile = Some(profile);
        self
    }

    /// Set the broker event to use.
    pub fn broker_event(mut self, event: &'a BrokerEvent) -> Self {
        self.event = Some(event);
        self
    }

    /// Create a [`Request::Trigger`] message.
    pub fn build_trigger(self) -> Result<Request, MessageError> {
        debug!("build_trigger: called");
        let profile = self.profile.ok_or(MessageError::NoProfile)?;
        debug!("build_trigger: got profile");
        let event = self.event.ok_or(MessageError::NoEvent)?;
        debug!("build_trigger: got event");

        let (rid, name, oid, old) = match event {
            BrokerEvent::Shutdown => panic!("got shutdown"),
            BrokerEvent::RefChanged {
                rid,
                name,
                oid,
                old,
            } => (rid, name, oid, old),
        };
        debug!("build_trigger: unpacked event");
        let is_patch = is_patch_update(name).is_some();
        let repository = profile.storage.repository(*rid)?;
        debug!("build_trigger: got repository");
        let storage = &profile.storage;
        let repo = radicle_surf::Repository::open(paths::repository(storage, rid))?;
        debug!("build_trigger: opened repository");
        let repo_project = repository.project()?;
        debug!("build_trigger: got project");
        let repo_identity = repository.identity()?;
        debug!("build_trigger: got identity");
        let author = match extract_author(profile, event) {
            Ok(author) => author,
            Err(err) => {
                debug!("build_trigger: author lookup failed: {err}");
                return Err(err);
            }
        };
        debug!("build_trigger: got author");
        let push_info: Option<PushEvent>;
        let patch_info: Option<PatchEvent>;
        let event_type: EventType;
        debug!("build_trigger: checking if patch or push");
        if is_patch {
            debug!("build_trigger: is patch");
            event_type = EventType::Patch;
            let patch_id = event.patch_id().ok_or(MessageError::Trigger)?;
            let patch = patch::Patches::open(&repository)?
                .get(&patch_id.into())?
                .ok_or(MessageError::Trigger)?;
            push_info = None;

            let revs: Vec<Revision> = patch
                .revisions()
                .map(|(rid, r)| {
                    Ok::<Revision, MessageError>(Revision {
                        id: rid.into(),
                        author: did_to_author(profile, r.author().id())?,
                        description: r.description().to_string(),
                        base: *r.base(),
                        oid: r.head(),
                        timestamp: r.timestamp().as_secs(),
                    })
                })
                .collect::<Result<Vec<Revision>, MessageError>>()?;
            let patch_author_pk = radicle::crypto::PublicKey::from(author.id);
            let patch_latest_revision = patch
                .latest_by(&patch_author_pk)
                .ok_or(MessageError::Trigger)?;
            let patch_head = patch_latest_revision.1.head();
            let patch_base = patch_latest_revision.1.base();
            let patch_commits: Vec<Oid> = repo
                .history(patch_head)?
                .take_while(|c| {
                    if let Ok(c) = c {
                        c.id != *patch_base
                    } else {
                        false
                    }
                })
                .map(|r| r.map(|c| c.id))
                .collect::<Result<Vec<Oid>, _>>()?;
            let patch_action = if patch.revisions().count() > 1 {
                "updated"
            } else {
                "created"
            };
            patch_info = Some(PatchEvent {
                action: PatchAction::try_from(patch_action)?,
                patch: Patch {
                    id: patch_id,
                    author,
                    title: patch.title().to_string(),
                    state: State {
                        status: patch.state().to_string(),
                        conflicts: match patch.state() {
                            patch::State::Open { conflicts, .. } => conflicts.to_vec(),
                            _ => vec![],
                        },
                    },
                    before: *patch_base,
                    after: patch_head,
                    commits: patch_commits,
                    target: patch.target().head(&repository)?,
                    labels: patch.labels().map(|l| l.name().to_string()).collect(),
                    assignees: patch.assignees().collect(),
                    revisions: revs,
                },
            });
        } else {
            debug!("build_trigger: is push");
            event_type = EventType::Push;
            let before_oid: Oid = old.unwrap_or(*oid);
            let push_commits: Vec<Oid> = repo
                .history(oid)?
                .take_while(|c| {
                    if let Ok(c) = c {
                        c.id != before_oid
                    } else {
                        false
                    }
                })
                .map(|r| r.map(|c| c.id))
                .collect::<Result<Vec<Oid>, _>>()?;
            push_info = Some(PushEvent {
                pusher: author,
                before: before_oid,
                after: *oid,
                branch: push_branch(name),
                commits: push_commits,
            });
            patch_info = None;
        };

        let common = EventCommonFields {
            version: PROTOCOL_VERSION,
            event_type,
            repository: Repository {
                id: *rid,
                name: repo_project.name().to_string(),
                description: repo_project.description().to_string(),
                private: !repo_identity.visibility.is_public(),
                default_branch: repo_project.default_branch().to_string(),
                delegates: repository.delegates()?.iter().copied().collect(),
            },
        };

        debug!("build_trigger: return Ok");
        Ok(Request::Trigger {
            common,
            push: push_info,
            patch: patch_info,
        })
    }
}

/// A request message sent by the broker to its adapter child process.
#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "request")]
#[serde(rename_all = "snake_case")]
#[non_exhaustive]
pub enum Request {
    /// Trigger a run.
    Trigger {
        /// Common fields for all message variants.
        #[serde(flatten)]
        common: EventCommonFields,

        /// The push event, if any.
        #[serde(flatten)]
        push: Option<PushEvent>,

        /// The patch event, if any.
        #[serde(flatten)]
        patch: Option<PatchEvent>,
    },
}

impl Request {
    /// Repository that the event concerns.
    pub fn repo(&self) -> RepoId {
        match self {
            Self::Trigger {
                common,
                push: _,
                patch: _,
            } => common.repository.id,
        }
    }

    /// Return the commit the event concerns. In other words, the
    /// commit that CI should run against.
    pub fn commit(&self) -> Oid {
        match self {
            Self::Trigger {
                common: _,
                push,
                patch,
            } => {
                if let Some(push) = push {
                    *push.commits.last().unwrap()
                } else if let Some(patch) = patch {
                    *patch.patch.commits.last().unwrap()
                } else {
                    panic!("neither push not panic: {self:#?}");
                }
            }
        }
    }

    /// Serialize the request as a single-line JSON, including the
    /// newline. This is meant for the broker to use.
    pub fn to_writer<W: Write>(&self, mut writer: W) -> Result<(), MessageError> {
        let mut line = serde_json::to_string(&self).map_err(MessageError::SerializeRequest)?;
        line.push('\n');
        writer
            .write(line.as_bytes())
            .map_err(MessageError::WriteRequest)?;
        Ok(())
    }

    /// Read a request from a reader. This is meant for the adapter to
    /// use.
    pub fn from_reader<R: Read>(reader: R) -> Result<Self, MessageError> {
        let mut line = String::new();
        let mut r = BufReader::new(reader);
        r.read_line(&mut line).map_err(MessageError::ReadLine)?;
        let req: Self =
            serde_json::from_slice(line.as_bytes()).map_err(MessageError::DeserializeRequest)?;
        Ok(req)
    }

    /// Parse a request from a string. This is meant for tests to use.
    pub fn try_from_str(s: &str) -> Result<Self, MessageError> {
        let req: Self =
            serde_json::from_slice(s.as_bytes()).map_err(MessageError::DeserializeRequest)?;
        Ok(req)
    }
}

fn did_to_author(profile: &Profile, did: &Did) -> Result<Author, MessageError> {
    let alias = profile.aliases().alias(did);
    Ok(Author { id: *did, alias })
}

fn extract_author(profile: &Profile, event: &BrokerEvent) -> Result<Author, MessageError> {
    debug!("extract_author: called");
    debug!("extract_author: {event:#?}");
    let nid = match event.nid() {
        Some(nid) => nid,
        None => {
            debug!("extract_author: nid lookup failed");
            return Err(MessageError::Trigger);
        }
    };
    debug!("extract_author: got nid {nid}");
    did_to_author(profile, &Did::from(nid))
}

impl fmt::Display for Request {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(
            f,
            "{}",
            serde_json::to_string(&self).map_err(|_| fmt::Error)?
        )
    }
}

/// Type of event.
#[derive(Debug, Clone, Copy, Eq, PartialEq, Serialize, Deserialize)]
#[serde(rename_all = "lowercase")]
pub enum EventType {
    /// A push event to a branch.
    Push,

    /// A new or changed patch.
    Patch,
}

/// Common fields in all variations of a [`Request`] message.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct EventCommonFields {
    /// Version of the request message.
    pub version: usize,

    /// The type of the event.
    pub event_type: EventType,

    /// The repository the event is related to.
    pub repository: Repository,
}

/// A push event.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct PushEvent {
    /// The author of the change.
    pub pusher: Author,

    /// The commit on which the change is based.
    pub before: Oid,

    /// FIXME
    pub after: Oid,

    /// The branch where the push occurred.
    pub branch: String,

    /// The commits in the change.
    pub commits: Vec<Oid>,
}

/// An event related to a Radicle patch object.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct PatchEvent {
    /// What action has happened to the patch.
    pub action: PatchAction,

    /// Metadata about the patch.
    pub patch: Patch,
}

/// What action has happened to the patch?
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum PatchAction {
    /// Patch has been created.
    Created,

    /// Patch has been updated.
    Updated,
}

#[cfg(test)]
impl PatchAction {
    fn as_str(&self) -> &str {
        match self {
            Self::Created => "created",
            Self::Updated => "updated",
        }
    }
}

impl TryFrom<&str> for PatchAction {
    type Error = MessageError;
    fn try_from(value: &str) -> Result<Self, Self::Error> {
        match value {
            "created" => Ok(Self::Created),
            "updated" => Ok(Self::Updated),
            _ => Err(Self::Error::UnknownPatchAction(value.into())),
        }
    }
}

/// Fields in a [`Request`] message describing the repository
/// concerned.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Repository {
    /// The unique repository id.
    pub id: RepoId,

    /// The name of the repository.
    pub name: String,

    /// A description of the repository.
    pub description: String,

    /// Is it a private repository?
    pub private: bool,

    /// The default branch in the repository: the branch that gets
    /// updated when a change is merged.
    pub default_branch: String,

    /// The delegates of the repository: those who can actually merge
    /// the change.
    pub delegates: Vec<Did>,
}

/// Fields describing the author of a change.
#[derive(Debug, Clone, Eq, PartialEq, Serialize, Deserialize)]
pub struct Author {
    /// The DID of the author. This is guaranteed to be unique.
    pub id: Did,

    /// The alias, or name, of the author. This need not be unique.
    pub alias: Option<Alias>,
}

impl std::fmt::Display for Author {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self.id)?;
        if let Some(alias) = &self.alias {
            write!(f, " ({})", alias)?;
        }
        Ok(())
    }
}

/// The state of a patch.
#[derive(Debug, Clone, Eq, PartialEq, Serialize, Deserialize)]
pub struct State {
    /// State of the patch.
    pub status: String,

    /// FIXME.
    pub conflicts: Vec<(RevisionId, Oid)>,
}

/// Revision of a patch.
#[derive(Debug, Clone, Eq, PartialEq, Serialize, Deserialize)]
pub struct Revision {
    /// FIXME.
    pub id: Oid,

    /// Author of the revision.
    pub author: Author,

    /// Description of the revision.
    pub description: String,

    /// Base commit on which the revision of the patch should be
    /// applied.
    pub base: Oid,

    /// FIXME.
    pub oid: Oid,

    /// Time stamp of the revision.
    pub timestamp: u64,
}

impl std::fmt::Display for Revision {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self.id)
    }
}

/// Metadata about a Radicle patch.
#[derive(Debug, Clone, Eq, PartialEq, Serialize, Deserialize)]
pub struct Patch {
    /// The patch id.
    pub id: Oid,

    /// The author of the patch.
    pub author: Author,

    /// The title of the patch.
    pub title: String,

    /// The state of the patch.
    pub state: State,

    /// The commit preceding the patch.
    pub before: Oid,

    /// FIXME.
    pub after: Oid,

    /// The list of commits in the patch.
    pub commits: Vec<Oid>,

    /// FIXME.
    pub target: Oid,

    /// Labels assigned to the patch.
    pub labels: Vec<String>,

    /// Who're in charge of the patch.
    pub assignees: Vec<Did>,

    /// List of revisions of the patch.
    pub revisions: Vec<Revision>,
}

/// A response message from the adapter child process to the broker.
#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(deny_unknown_fields)]
#[serde(rename_all = "snake_case")]
#[serde(tag = "response")]
pub enum Response {
    /// A CI run has been triggered.
    Triggered {
        /// The identifier for the CI run assigned by the adapter.
        run_id: RunId,

        /// Optional informational URL for the run.
        info_url: Option<String>,
    },

    /// A CI run has finished.
    Finished {
        /// The result of a CI run.
        result: RunResult,
    },
}

impl Response {
    /// Create a `Response::Triggered` message without an info URL.
    pub fn triggered(run_id: RunId) -> Self {
        Self::Triggered {
            run_id,
            info_url: None,
        }
    }

    /// Create a `Response::Triggered` message with an info URL.
    pub fn triggered_with_url(run_id: RunId, url: &str) -> Self {
        Self::Triggered {
            run_id,
            info_url: Some(url.into()),
        }
    }

    /// Create a `Response::Finished` message.
    pub fn finished(result: RunResult) -> Self {
        Self::Finished { result }
    }

    /// Does the message indicate a result for the CI run?
    pub fn result(&self) -> Option<&RunResult> {
        if let Self::Finished { result } = self {
            Some(result)
        } else {
            None
        }
    }

    /// Serialize a response as a single-line JSON, including the
    /// newline. This is meant for the adapter to use.
    pub fn to_writer<W: Write>(&self, mut writer: W) -> Result<(), MessageError> {
        let mut line = serde_json::to_string(&self).map_err(MessageError::SerializeResponse)?;
        line.push('\n');
        writer
            .write(line.as_bytes())
            .map_err(MessageError::WriteResponse)?;
        Ok(())
    }

    /// Read a response from a reader. This is meant for the broker to
    /// use.
    pub fn from_reader<R: Read + BufRead>(reader: &mut R) -> Result<Option<Self>, MessageError> {
        let mut line = String::new();
        let mut r = BufReader::new(reader);
        let n = r.read_line(&mut line).map_err(MessageError::ReadLine)?;
        if n == 0 {
            // Child's stdout was closed.
            Ok(None)
        } else {
            let req: Self = serde_json::from_slice(line.as_bytes())
                .map_err(MessageError::DeserializeResponse)?;
            Ok(Some(req))
        }
    }

    /// Read a response from a string slice. This is meant for the
    /// broker to use.
    #[allow(clippy::should_implement_trait)]
    pub fn from_str(line: &str) -> Result<Self, MessageError> {
        let req: Self =
            serde_json::from_slice(line.as_bytes()).map_err(MessageError::DeserializeResponse)?;
        Ok(req)
    }
}

/// All possible errors from the CI broker messages.
#[derive(Debug, thiserror::Error)]
pub enum MessageError {
    /// [`RequestBuilder`] does not have profile set.
    #[error("RequestBuilder must have profile set")]
    NoProfile,

    /// [`RequestBuilder`] does not have event set.
    #[error("RequestBuilder must have broker event set")]
    NoEvent,

    /// Failed to serialize a request message as JSON. This should
    /// never happen and likely indicates a programming failure.
    #[error("failed to serialize a request into JSON to a file handle")]
    SerializeRequest(#[source] serde_json::Error),

    /// Failed to serialize a response message as JSON. This should never
    /// happen and likely indicates a programming failure.
    #[error("failed to serialize a request into JSON to a file handle")]
    SerializeResponse(#[source] serde_json::Error),

    /// Failed to write the serialized request message to an open file.
    #[error("failed to write JSON to file handle")]
    WriteRequest(#[source] std::io::Error),

    /// Failed to write the serialized response message to an open
    /// file.
    #[error("failed to write JSON to file handle")]
    WriteResponse(#[source] std::io::Error),

    /// Failed to read a line of JSON from an open file.
    #[error("failed to read line from file handle")]
    ReadLine(#[source] std::io::Error),

    /// Failed to parse JSON as a request or a response.
    #[error("failed to read a JSON request from a file handle")]
    DeserializeRequest(#[source] serde_json::Error),

    /// Failed to parse JSON as a response or a response.
    #[error("failed to read a JSON response from a file handle")]
    DeserializeResponse(#[source] serde_json::Error),

    /// Error from Radicle.
    #[error(transparent)]
    RadicleProfile(#[from] radicle::profile::Error),

    /// Error retrieving context to generate trigger from BrokerEvent
    #[error("could not generate trigger from event")]
    Trigger,

    /// Error from Radicle storage.
    #[error(transparent)]
    StorageError(#[from] radicle::storage::Error),

    /// Error from Radicle repository.
    #[error(transparent)]
    RepositoryError(#[from] radicle::storage::RepositoryError),

    /// Error from Radicle COB.
    #[error(transparent)]
    CobStoreError(#[from] radicle::cob::store::Error),

    /// Error from `radicle-surf` crate.
    #[error(transparent)]
    RadicleSurfError(#[from] radicle_surf::Error),

    /// Trying to create a PatchAction from an invalid value.
    #[error("invalid patch action {0:?}")]
    UnknownPatchAction(String),
}

#[cfg(test)]
pub mod tests {
    use crate::event::BrokerEvent;
    use crate::msg::{EventType, Request, RequestBuilder};
    use radicle::git::raw::Oid;
    use radicle::git::RefString;
    use radicle::patch::{MergeTarget, Patches};
    use radicle::prelude::Did;
    use radicle::storage::ReadRepository;

    use crate::test::{MockNode, TestResult};

    #[test]
    fn trigger_push() -> TestResult<()> {
        let mock_node = MockNode::new()?;
        let profile = mock_node.profile()?;

        let project = mock_node.node().project();
        let (_, repo_head) = project.repo.head()?;
        let cmt = radicle::test::fixtures::commit(
            "my test commit",
            &[repo_head.into()],
            &project.backend,
        );

        let be = BrokerEvent::RefChanged {
            rid: project.id,
            name: RefString::try_from(
                "refs/namespaces/$nid/refs/heads/master".replace("$nid", &profile.id().to_string()),
            )?,
            oid: cmt,
            old: Some(repo_head),
        };

        let req = RequestBuilder::default()
            .profile(&profile)
            .broker_event(&be)
            .build_trigger()
            .unwrap();
        let Request::Trigger {
            common,
            push,
            patch,
        } = req;

        assert!(patch.is_none());
        assert!(push.is_some());
        assert_eq!(common.event_type, EventType::Push);
        assert_eq!(common.repository.id, project.id);
        assert_eq!(common.repository.name, project.repo.project()?.name());

        let push = push.unwrap();
        assert_eq!(push.after, cmt);
        assert_eq!(push.before, repo_head);
        assert_eq!(
            push.branch,
            "master".replace("$nid", &profile.id().to_string())
        );
        assert_eq!(push.commits, vec![cmt]);
        assert_eq!(push.pusher.id, Did::from(profile.id()));

        Ok(())
    }

    #[test]
    fn trigger_patch() -> TestResult<()> {
        let mock_node = MockNode::new()?;
        let profile = mock_node.profile()?;

        let project = mock_node.node().project();
        let (_, repo_head) = project.repo.head()?;
        let cmt = radicle::test::fixtures::commit(
            "my test commit",
            &[repo_head.into()],
            &project.backend,
        );

        let node = mock_node.node();

        let mut patches = Patches::open(&project.repo)?;
        let mut cache = radicle::cob::cache::NoCache;
        let patch_cob = patches.create(
            "my patch title",
            "my patch description",
            MergeTarget::Delegates,
            repo_head,
            cmt,
            &[],
            &mut cache,
            &node.signer,
        )?;

        let be = BrokerEvent::RefChanged {
            rid: project.id,
            name: RefString::try_from(
                "refs/namespaces/$nid/refs/cobs/xyz.radicle.patch/$patchId"
                    .replace("$nid", &profile.id().to_string())
                    .replace("$patchId", &patch_cob.id.to_string()),
            )?,
            oid: radicle_git_ext::Oid::from(Oid::from_str(&patch_cob.id.to_string())?),
            old: None,
        };

        let req = RequestBuilder::default()
            .profile(&profile)
            .broker_event(&be)
            .build_trigger()
            .unwrap();
        let Request::Trigger {
            common,
            push,
            patch,
        } = req;

        assert!(patch.is_some());
        assert!(push.is_none());
        assert_eq!(common.event_type, EventType::Patch);
        assert_eq!(common.repository.id, project.id);
        assert_eq!(common.repository.name, project.repo.project()?.name());

        let patch = patch.unwrap();
        assert_eq!(patch.action.as_str(), "created");
        assert_eq!(patch.patch.id.to_string(), patch_cob.id.to_string());
        assert_eq!(patch.patch.title, patch_cob.title());
        assert_eq!(patch.patch.state.status, patch_cob.state().to_string());
        assert_eq!(patch.patch.target, repo_head);
        assert_eq!(patch.patch.revisions.len(), 1);
        let rev = patch.patch.revisions.first().unwrap();
        assert_eq!(rev.id.to_string(), patch_cob.id.to_string());
        assert_eq!(rev.base, repo_head);
        assert_eq!(rev.oid, cmt);
        assert_eq!(rev.author.id, Did::from(profile.id()));
        assert_eq!(rev.description, patch_cob.description());
        assert_eq!(rev.timestamp, patch_cob.timestamp().as_secs());
        assert_eq!(patch.patch.after, cmt);
        assert_eq!(patch.patch.before, repo_head);
        assert_eq!(patch.patch.commits, vec![cmt]);

        Ok(())
    }
}
