//! A logger abstraction for the CI broker.

use std::{path::Path, process::ExitStatus, time::Duration};

use clap::ValueEnum;
use radicle::{git::raw::Oid, identity::RepoId, node::Event};
use serde_json::Value;
use tracing::{debug, error, info, trace, warn, Level};
use tracing_subscriber::{fmt, layer::SubscriberExt, util::SubscriberInitExt, EnvFilter};

use crate::{
    ci_event::CiEvent,
    ci_event_source::CiEventSource,
    config::Config,
    db::{QueueId, QueuedCiEvent},
    msg::Request,
    node_event_source::NodeEventSource,
    queueproc::QueueError,
    run::Run,
};

#[derive(Debug, thiserror::Error)]
pub enum KindError {
    #[error("unknown log message kind {0:?}")]
    Unknown(String),

    #[error("unknown log message kind {0:?}")]
    UnknownValue(Value),
}

/// A unique type or kind for each log message.
#[derive(Debug, Clone, ValueEnum, Eq, PartialEq)]
pub enum Kind {
    Debug,
    Startup,
    Shutdown,
    GotEvent,
    StartRun,
    AdapterMessage,
    FinishRun,
}

impl std::fmt::Display for Kind {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let s = match self {
            Self::Debug => "debug",
            Self::Startup => "startup",
            Self::Shutdown => "shutdown",
            Self::GotEvent => "got_event",
            Self::StartRun => "start_run",
            Self::FinishRun => "finish_run",
            Self::AdapterMessage => "adapter_message",
        };

        write!(f, "{s}")
    }
}

impl TryFrom<&str> for Kind {
    type Error = KindError;

    fn try_from(value: &str) -> Result<Self, Self::Error> {
        match value {
            "debug" => Ok(Self::Debug),
            "startup" => Ok(Self::Startup),
            "shutdown" => Ok(Self::Shutdown),
            "got_event" => Ok(Self::GotEvent),
            "start_run" => Ok(Self::StartRun),
            "finish_run" => Ok(Self::FinishRun),
            "adapter_message" => Ok(Self::AdapterMessage),
            _ => Err(KindError::Unknown(value.into())),
        }
    }
}

impl TryFrom<Value> for Kind {
    type Error = KindError;

    fn try_from(value: Value) -> Result<Self, Self::Error> {
        match value {
            Value::String(s) => match s.as_str() {
                "debug" => Ok(Self::Debug),
                "startup" => Ok(Self::Startup),
                "shutdown" => Ok(Self::Shutdown),
                "got_event" => Ok(Self::GotEvent),
                "start_run" => Ok(Self::StartRun),
                "finish_run" => Ok(Self::FinishRun),
                "adapter_message" => Ok(Self::AdapterMessage),
                _ => Err(KindError::Unknown(s)),
            },
            _ => Err(KindError::UnknownValue(value)),
        }
    }
}

// An identifier for the specific log message, to avoid having to
// match on message text.
#[derive(Debug, Copy, Clone)]
enum Id {
    AdapterConfig,
    AdapterExitCode,
    AdapterInvoluntaryExit,
    AdapterNoExit,
    AdapterNoFirstMessage,
    AdapterNoSecondMessage,
    AdapterStderrLine,
    AdapterTooManyMessages,

    AdHoc,

    BrokerDatabase,
    BrokerRunEnd,
    BrokerRunStart,
    BrokerTriggerMessage,

    CiEventSourceCreated,
    CiEventSourceDisconnected,
    CiEventSourceEnd,
    CiEventSourceEndOfFile,
    CiEventSourceGotEvents,
    CibConfig,
    CibEndFailure,
    CibEndSuccess,
    CibStart,

    NodeEventSourceCreated,
    NodeEventSourceDisconnected,
    NodeEventSourceEnd,
    NodeEventSourceEndOfFile,
    NodeEventSourceReceivedEvent,

    PagesDisconnected,
    PagesEnd,
    PagesInterval,
    PagesNoDirSet,
    PagesStart,

    QueueAddEnd,
    QueueAddEndEvents,
    QueueAddEnqueueEvent,
    QueueAddStart,

    QueueProcActionRun,
    QueueProcActionShutdown,
    QueueProcDisconnected,
    QueueProcEnd,
    QueueProcPickedEvent,
    QueueProcProcessedEvent,
    QueueProcQueueLength,
    QueueProcRemoveEvent,
    QueueProcStart,

    TimeoutLineReceiverCheckChild,
    TimeoutLineReceiverChildDisconnected,
    TimeoutLineReceiverDisconnected,
    TimeoutLineReceiverLine,
    TimeoutLineReceiverLostChild,
    TimeoutLineReceiverTried,
    TimeoutLineReceiverTry,
    TimeoutNannyEnd,
    TimeoutNannyLostChild,
    TimeoutNannyNotifyTooLong,
    TimeoutNannyStart,
    TimeoutNannyTooLong,
    TimeoutNannyWaitEnd,
    TimeoutNannyWaitWord,
    TimeoutNannyWord,
    TimeoutNonblockingEnd,
    TimeoutNonblockingEndOfFile,
    TimeoutNonblockingError,
    TimeoutNonblockingTooMuch,
    TimeoutNonblockingTried,
    TimeoutNonblockingTry,
    TimeoutOk,
    TimeoutRequestEnd,
    TimeoutWaitChildEnd,
    TimeoutWaitChildExitStatus,
    TimeoutWaitStderrReaderEnd,
    TimeoutWaitStdinWriterEnd,
    TimeoutWaitStdoutReaderEnd,
}

#[derive(Debug, thiserror::Error)]
pub enum LogLevelError {
    #[error("unknown log level {0}")]
    UnknownLogLevel(String),

    #[error("unknown log level {0:?}")]
    UnknownLogLevelValue(Value),
}

// We define our own type for log levels so that we can apply
// clap::ValueEnum on it.
#[derive(ValueEnum, Ord, PartialOrd, Eq, PartialEq, Copy, Clone, Debug)]
pub enum LogLevel {
    Trace,
    Debug,
    Info,
    Warning,
    Error,
}

impl TryFrom<&str> for LogLevel {
    type Error = LogLevelError;
    fn try_from(s: &str) -> Result<Self, LogLevelError> {
        match s {
            "TRACE" => Ok(Self::Trace),
            "DEBUG" => Ok(Self::Debug),
            "INFO" => Ok(Self::Info),
            "WARNING" => Ok(Self::Warning),
            "ERROR" => Ok(Self::Error),
            _ => Err(LogLevelError::UnknownLogLevel(s.into())),
        }
    }
}

impl TryFrom<String> for LogLevel {
    type Error = LogLevelError;
    fn try_from(s: String) -> Result<Self, LogLevelError> {
        Self::try_from(s.as_str())
    }
}

impl TryFrom<&Value> for LogLevel {
    type Error = LogLevelError;
    fn try_from(v: &Value) -> Result<Self, LogLevelError> {
        match v {
            Value::String(s) => Self::try_from(s.as_str()),
            _ => Err(LogLevelError::UnknownLogLevelValue(v.clone())),
        }
    }
}

impl TryFrom<Value> for LogLevel {
    type Error = LogLevelError;
    fn try_from(v: Value) -> Result<Self, LogLevelError> {
        Self::try_from(&v)
    }
}

impl std::fmt::Display for LogLevel {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let s = match self {
            Self::Trace => "trace",
            Self::Debug => "debug",
            Self::Info => "info",
            Self::Warning => "warn",
            Self::Error => "error",
        };

        write!(f, "{s}")
    }
}

impl From<LogLevel> for tracing::Level {
    fn from(log_level: LogLevel) -> Self {
        match log_level {
            LogLevel::Trace => Level::TRACE,
            LogLevel::Debug => Level::DEBUG,
            LogLevel::Info => Level::INFO,
            LogLevel::Warning => Level::WARN,
            LogLevel::Error => Level::ERROR,
        }
    }
}

#[allow(clippy::unwrap_used)]
pub fn open(level: LogLevel) {
    #[cfg(test)]
    let writer = fmt::TestWriter::new();

    #[cfg(not(test))]
    let writer = std::io::stderr;

    let fmt_layer = fmt::layer().with_target(false).with_writer(writer).json();

    let filter_layer = EnvFilter::try_new(level.to_string()).unwrap();

    tracing_subscriber::registry()
        .with(filter_layer)
        .with(fmt_layer)
        .init();
}

pub fn start_cib() {
    info!(
        msg_id = ?Id::CibStart,
        kind = %Kind::Startup,
        version = env!("GIT_HEAD"),
        "CI broker starts"
    );
}

pub fn end_cib_successfully() {
    info!(
        msg_id = ?Id::CibEndSuccess,
        kind = %Kind::Shutdown,
        success = true,
        "CI broker ends successfully"
    );
}

pub fn end_cib_in_error() {
    error!(
        msg_id = ?Id::CibEndFailure,
        kind = %Kind::Shutdown,
        success = false,
        "CI broker ends in unrecoverable error"
    );
}

pub fn node_event_source_created(source: &NodeEventSource) {
    debug!(
        msg_id = ?Id::NodeEventSourceCreated,
        kind = %Kind::Startup,
        node_event_source = ?source,
        "created node event source"
    );
}

pub fn node_event_source_got_event(event: &Event) {
    trace!(
        msg_id = ?Id::NodeEventSourceReceivedEvent,
        kind = %Kind::GotEvent,
        node_event = ?event,
        "node event source received event"
    );
}

pub fn node_event_source_eof(source: &NodeEventSource) {
    debug!(
        msg_id = ?Id::NodeEventSourceEndOfFile,
        kind = %Kind::Shutdown,
        node_event_source = ?source,
        "node event source end of file on control socket"
    );
}

pub fn ci_event_source_created(source: &CiEventSource) {
    debug!(
        msg_id = ?Id::CiEventSourceCreated,
        kind = %Kind::Startup,
        ?source,
        "created CI event source"
    );
}

pub fn ci_event_source_got_events(events: &[CiEvent]) {
    trace!(
        msg_id = ?Id::CiEventSourceGotEvents,
        kind = %Kind::GotEvent,
        ?events,
        "CI event source received events"
    );
}

pub fn ci_event_source_disconnected() {
    info!(
        msg_id = ?Id::CiEventSourceDisconnected,
        kind = %Kind::Shutdown,
        "CI event source received disconnection"
    );
}

pub fn ci_event_source_end() {
    info!(
        msg_id = ?Id::CiEventSourceEnd,
        kind = %Kind::Debug,
        "CI event source was notified end of events"
    );
}

pub fn ci_event_source_eof(source: &CiEventSource) {
    info!(
        msg_id = ?Id::CiEventSourceEndOfFile,
        kind = %Kind::Debug,
        ?source,
        "CI event source end of file"
    );
}

pub fn loaded_config(config: &Config) {
    debug!(
        msg_id = ?Id::CibConfig,
        kind = %Kind::Startup,
        ?config,
        "loaded configuration"
    );
}

pub fn adapter_config(config: &Config) {
    trace!(
        msg_id = ?Id::AdapterConfig,
        kind = %Kind::Debug,
        ?config,
        "adapter configuration"
    );
}

pub fn queueproc_start() {
    debug!(
        msg_id = ?Id::QueueProcStart,
        kind = %Kind::Startup,
        "start thread to process events until a shutdown event"
    );
}

pub fn queueproc_end() {
    debug!(
        msg_id = ?Id::QueueProcEnd,
        kind = %Kind::Debug,
        "thread to process events ends"
    );
}

pub fn queueproc_channel_disconnect() {
    info!(
        msg_id = ?Id::QueueProcDisconnected,
        kind = %Kind::Debug,
        "event notification channel disconnected"
    );
}

pub fn queueproc_queue_length(len: usize) {
    trace!(
        msg_id = ?Id::QueueProcQueueLength,
        kind = %Kind::Debug,
        ?len,
        "event queue length"
    );
}

pub fn queueproc_picked_event(id: &QueueId, event: &QueuedCiEvent) {
    info!(
        msg_id = ?Id::QueueProcPickedEvent,
        kind = %Kind::GotEvent,
        ?id,
        ?event,
        "picked event from queue"
    );
}

pub fn queueproc_processed_event(result: &Result<bool, QueueError>) {
    info!(
        msg_id = ?Id::QueueProcProcessedEvent,
        kind = %Kind::GotEvent,
        ?result,
        "result of processing event"
    );
}

pub fn queueproc_remove_event(id: &QueueId) {
    info!(
        msg_id = ?Id::QueueProcRemoveEvent,
        kind = %Kind::Debug,
        ?id,
        "remove event from queue"
    );
}

pub fn queueproc_action_run(rid: &RepoId, oid: &Oid) {
    info!(
        msg_id = ?Id::QueueProcActionRun,
        kind = %Kind::Debug,
        ?rid,
        ?oid,
        "Action: run"
    );
}

pub fn queueproc_action_shutdown() {
    info!(
        msg_id = ?Id::QueueProcActionShutdown,
        kind = %Kind::Debug,
        "Action: shutdown"
    );
}

pub fn queueadd_start() {
    debug!(
        msg_id = ?Id::QueueAddStart,
        kind = %Kind::Debug,
        "start thread to add events from node to event queue"
    );
}

pub fn queueadd_control_socket_close() {
    info!(
        msg_id = ?Id::QueueAddEndEvents,
        kind = %Kind::Debug,
        "no more events from node control socket"
    );
}

pub fn queueadd_push_event(event: &CiEvent) {
    debug!(
        msg_id = ?Id::QueueAddEnqueueEvent,
        kind = %Kind::GotEvent,
        ?event,
        "insert broker event into queue"
    );
}

pub fn queueadd_end() {
    debug!(
        msg_id = ?Id::QueueAddEnd,
        kind = %Kind::Debug,
        "thread to process events ends"
    );
}

pub fn pages_directory_unset() {
    warn!(
        msg_id = ?Id::PagesNoDirSet,
        kind = %Kind::Debug,
        "not writing HTML report pages as output directory has not been set"
    );
}

pub fn pages_interval(interval: Duration) {
    trace!(
        msg_id = ?Id::PagesInterval,
        kind = %Kind::Debug,
        interval = interval.as_secs(),
        "interval for waiting between HTML page updates",
    );
}

pub fn pages_disconnected() {
    info!(
        msg_id = ?Id::PagesDisconnected,
        kind = %Kind::Debug,
        "page updater: run notification channel disconnected"
    );
}

pub fn pages_start() {
    debug!(
        msg_id = ?Id::PagesStart,
        kind = %Kind::Debug,
        "start page updater thread"
    );
}

pub fn pages_end() {
    debug!(
        msg_id = ?Id::PagesEnd,
        kind = %Kind::Debug,
        "end page updater thread"
    );
}

pub fn event_disconnected() {
    info!(
        msg_id = ?Id::NodeEventSourceDisconnected,
        kind = %Kind::Debug,
        "connection to node control socket broke"
    );
}

pub fn event_end() {
    info!(
        msg_id = ?Id::NodeEventSourceEnd,
        kind = %Kind::Debug,
        "no more node events from control socket: iterator ended"
    );
}

pub fn broker_db(filename: &Path) {
    info!(
        msg_id = ?Id::BrokerDatabase,
        kind = %Kind::Startup,
        filename = %filename.display(),
        "broker database"
    );
}

pub fn broker_start_run(trigger: &Request) {
    info!(
        msg_id = ?Id::BrokerRunStart,
        kind = %Kind::StartRun,
        "start CI run"
    );
    debug!(
        msg_id = ?Id::BrokerTriggerMessage,
        kind = %Kind::Debug,
        event = ?trigger,
        "trigger message"
    );
}

pub fn broker_end_run(run: &Run) {
    info!(
        msg_id = ?Id::BrokerRunEnd,
        kind = %Kind::FinishRun,
        ?run,
        "Finish CI run"
    );
}

pub fn adapter_no_first_response() {
    error!(
        msg_id = ?Id::AdapterNoFirstMessage,
        kind = %Kind::AdapterMessage,
        "no first response message"
    );
}

pub fn adapter_no_second_response() {
    error!(
        msg_id = ?Id::AdapterNoSecondMessage,
        kind = %Kind::AdapterMessage,
        "no second response message"
    );
}

pub fn adapter_too_many_responses() {
    error!(
        msg_id = ?Id::AdapterTooManyMessages,
        kind = %Kind::AdapterMessage,
        "too many response messages"
    );
}

pub fn adapter_stderr_line(line: &str) {
    debug!(
        msg_id = ?Id::AdapterStderrLine,
        kind = %Kind::Debug,
        stderr_line = line,
        "adapter stderr"
    );
}

pub fn adapter_result(exit: i32) {
    debug!(
        msg_id = ?Id::AdapterExitCode,
        kind = %Kind::Debug,
        exit_code = exit,
        "adapter exit code"
    );
}

pub fn adapter_did_not_exit_voluntarily() {
    warn!(
        msg_id = ?Id::AdapterInvoluntaryExit,
        kind = %Kind::Debug,
        "adapter did not exit voluntarily: terminated for taking too long"
    );
}

pub fn adapter_did_not_exit() {
    warn!(
        msg_id = ?Id::AdapterNoExit,
        kind = %Kind::Debug,
        "adapter did not exit: probably killed by signal"
    );
}

pub fn timeoutcmd_request_termination(result: Result<(), std::sync::mpsc::SendError<()>>) {
    trace!(
        msg_id = ?Id::TimeoutRequestEnd,
        kind = %Kind::Debug,
        ?result,
        "request termination of child process"
    );
}

pub fn timeoutcmd_wait_word_from_nanny() {
    trace!(
        msg_id = ?Id::TimeoutNannyWaitWord,
        kind = %Kind::Debug,
        "wait: wait for word from nanny"
    );
}

pub fn timeoutcmd_wait_got_word_from_nanny() {
    trace!(
        msg_id = ?Id::TimeoutNannyWord,
        kind = %Kind::Debug,
        "got word from nanny"
    );
}

pub fn timeoutcmd_wait_on_nanny_to_end() {
    trace!(
        msg_id = ?Id::TimeoutNannyWaitEnd,
        kind = %Kind::Debug,
        "wait: wait on nanny thread to end"
    );
}

pub fn timeoutcmd_wait_on_stdin_writer_to_end() {
    trace!(
        msg_id = ?Id::TimeoutWaitStdinWriterEnd,
        kind = %Kind::Debug,
        "wait: wait for stdin writer to terminate"
    );
}

pub fn timeoutcmd_wait_on_stdout_reader_to_end() {
    trace!(
        msg_id = ?Id::TimeoutWaitStdoutReaderEnd,
        kind = %Kind::Debug,
        "wait: wait for stdout reader to terminate"
    );
}

pub fn timeoutcmd_wait_on_stderr_reader_to_end() {
    trace!(
        msg_id = ?Id::TimeoutWaitStderrReaderEnd,
        kind = %Kind::Debug,
        "wait: wait for stderr reader to terminate"
    );
}

pub fn timeoutcmd_wait_on_child_to_end() {
    trace!(
        msg_id = ?Id::TimeoutWaitChildEnd,
        lkind = %Kind::Debug,
        "wait: wait for child to terminate"
    );
}

pub fn timeoutcmd_wait_status(status: ExitStatus) {
    trace!(
        msg_id = ?Id::TimeoutWaitChildExitStatus,
        kind = %Kind::Debug,
        ?status,
        "wait: wait status"
    );
}

pub fn timeoutcmd_ok() {
    trace!(
        msg_id = ?Id::TimeoutOk,
        kind = %Kind::Debug,
        "wait: return Ok result"
    );
}

pub fn timeoutcmd_nanny_start() {
    trace!(
        kind = %Kind::Debug,
        "nanny: start monitoring child"
    );
}

pub fn timeoutcmd_nanny_terminated_as_requested(result: Result<(), std::io::Error>) {
    trace!(
        msg_id = ?Id::TimeoutNannyStart,
        kind = %Kind::Debug,
        ?result,
        "nanny: terminated child by request"
    );
}

pub fn timeoutcmd_nanny_too_long(
    id: u32,
    elapsed: Duration,
    max: Duration,
    result: Result<(), std::io::Error>,
) {
    trace!(
        msg_id = ?Id::TimeoutNannyTooLong,
        kind = %Kind::Debug,
        %id,
        elapsed_ms = elapsed.as_millis(),
        max_ms = max.as_millis(),
        ?result,
        "nanny: child has run for too long",
    );
}

pub fn timeoutcmd_nanny_child_died() {
    trace!(
        msg_id = ?Id::TimeoutNannyLostChild,
        kind = %Kind::Debug,
        "nanny: child has terminated"
    );
}

pub fn timeoutcmd_nanny_time_to_end() {
    trace!(
        msg_id = ?Id::TimeoutNannyNotifyTooLong,
        kind = %Kind::Debug,
        "nanny: tell other threads it's time to end"
    );
}

pub fn timeoutcmd_nanny_ends() {
    trace!(
        msg_id = ?Id::TimeoutNannyEnd,
        kind = %Kind::Debug,
        "nanny: ends"
    );
}

pub fn timeoutcmd_line_reader_try_byte(thread: &'static str) {
    trace!(
        msg_id = ?Id::TimeoutLineReceiverTry,
        kind = %Kind::Debug,
        %thread,
        "line receiver: try to receive next byte"
    );
}

pub fn timeoutcmd_line_reader_tried_byte(
    thread: &'static str,
    result: Result<u8, std::sync::mpsc::TryRecvError>,
) {
    trace!(
        msg_id = ?Id::TimeoutLineReceiverTried,
        kind = %Kind::Debug,
        ?result,
        %thread,
        "line receiver: tried to read line"
    );
}

pub fn timeoutcmd_line_reader_got_line(thread: &'static str, line: &str) {
    trace!(
        msg_id = ?Id::TimeoutLineReceiverLine,
        kind = %Kind::Debug,
        ?line,
        %thread,
        "line-receiver: received line"
    );
}

pub fn timeoutcmd_line_reader_got_disconnected(thread: &'static str) {
    trace!(
        msg_id = ?Id::TimeoutLineReceiverDisconnected,
        kind = %Kind::Debug,
        %thread,
        "line-receiver: disconnected"
    );
}

pub fn timeoutcmd_line_reader_did_child_die(thread: &'static str) {
    trace!(
        msg_id = ?Id::TimeoutLineReceiverCheckChild,
        kind = %Kind::Debug,
        %thread,
        "line-receiver: has child terminated?"
    );
}

pub fn timeoutcmd_line_reader_child_died(thread: &'static str) {
    trace!(
        msg_id = ?Id::TimeoutLineReceiverLostChild,
        kind = %Kind::Debug,
        %thread,
        "line receiver: OK: child has terminated, not returning line",
    );
}

pub fn timeoutcmd_line_reader_child_channel_disconnected(thread: &'static str) {
    trace!(
        msg_id = ?Id::TimeoutLineReceiverChildDisconnected,
        kind = %Kind::Debug,
        %thread,
        "line receiver: Disconnected: child has terminated, not returning line",
    );
}

pub fn timeoutcmd_nonblocking_try_byte(thread: &'static str, count: usize) {
    trace!(
        msg_id = ?Id::TimeoutNonblockingTry,
        kind = %Kind::Debug,
        received_so_far = count,
        %thread,
        "read_to_end: try to receive next byte",
    );
}

pub fn timeoutcmd_nonblocking_tried_byte(
    thread: &'static str,
    result: &Result<usize, std::io::Error>,
    byte: &[u8],
) {
    trace!(
        msg_id = ?Id::TimeoutNonblockingTried,
        kind = %Kind::Debug,
        ?result,
        ?byte,
        %thread,
        "read_to_end: tried to receive byte",
    );
}

pub fn timeoutcmd_nonblocking_eof(thread: &'static str) {
    trace!(
        msg_id = ?Id::TimeoutNonblockingEndOfFile,
        kind = %Kind::Debug,
        %thread,
        "read_to_end: got end of file"
    );
}

pub fn timeoutcmd_nonblocking_got_too_much(
    thread: &'static str,
    result: Result<usize, std::io::Error>,
    byte: &[u8],
) {
    trace!(
        msg_id = ?Id::TimeoutNonblockingTooMuch,
        kind = %Kind::Debug,
        ?result,
        ?byte,
        %thread,
        "read_to_end: received too much",
    );
}

pub fn timeoutcmd_nonblocking_read_error(thread: &'static str, err: &std::io::Error) {
    trace!(
        msg_id = ?Id::TimeoutNonblockingError,
        kind = %Kind::Debug,
        ?err,
        %thread,
        "read_to_end: read error"
    );
}

pub fn timeoutcmd_nonblocking_ends(thread: &'static str) {
    trace!(
        msg_id = ?Id::TimeoutNonblockingEnd,
        kind = %Kind::Debug,
        %thread,
        "read_to_end: ends"
    );
}

pub fn debug(msg: &str) {
    debug!(
        msg_id = ?Id::AdHoc,
        kind = %Kind::Debug,
        "{msg}"
    );
}

pub fn debug2(msg: String) {
    debug!(
        msg_id = ?Id::AdHoc,
        kind = %Kind::Debug,
        "{msg}"
    );
}

pub fn trace(msg: &str) {
    trace!(
        msg_id = ?Id::AdHoc,
        kind = %Kind::Debug,
        "{msg}"
    );
}

pub fn trace2(msg: String) {
    trace!(
        msg_id = ?Id::AdHoc,
        kind = %Kind::Debug,
        "{msg}"
    );
}

pub fn error(msg: &str, e: &impl std::error::Error) {
    error!("{msg}: {e}");
    let mut e = e.source();
    while let Some(source) = e {
        error!("caused by: {}", source);
        e = source.source();
    }
}
