//! The CI broker.

#![allow(clippy::result_large_err)]

use std::{
    error::Error,
    fs::write,
    path::{Path, PathBuf},
    process::exit,
};

use clap::Parser;
use log::{debug, error, info};

use radicle::Profile;

use radicle_ci_broker::{
    adapter::Adapter,
    broker::{Broker, BrokerError},
    config::{Config, ConfigError},
    db::{Db, DbError},
    pages::{PageBuilder, PageError},
    queueadd::{AdderError, QueueAdderBuilder},
    queueproc::{QueueError, QueueProcessorBuilder},
};

fn main() {
    if let Err(e) = fallible_main() {
        error!("ERROR: {}", e);
        let mut e = e.source();
        while let Some(source) = e {
            error!("caused by: {}", source);
            e = source.source();
        }
        exit(1);
    }
}

fn fallible_main() -> Result<(), CibError> {
    let args = Args::parse();

    pretty_env_logger::init_custom_env("RADICLE_CI_BROKER_LOG");
    info!("Radicle CI broker starts");

    let config = Config::load(&args.config).map_err(|e| CibError::read_config(&args.config, e))?;
    debug!("loaded configuration: {:#?}", config);

    args.run(&config)?;

    Ok(())
}

#[derive(Debug, Parser)]
struct Args {
    #[clap(long)]
    config: PathBuf,

    #[clap(subcommand)]
    cmd: Cmd,
}

impl Args {
    fn run(&self, config: &Config) -> Result<(), CibError> {
        match &self.cmd {
            Cmd::Config(x) => x.run(self, config)?,
            Cmd::Insert(x) => x.run(self, config)?,
            Cmd::Queued(x) => x.run(self, config)?,
            Cmd::ProcessEvents(x) => x.run(self, config)?,
        }
        Ok(())
    }

    fn open_db(&self, config: &Config) -> Result<Db, CibError> {
        Db::new(&config.db).map_err(CibError::db)
    }
}

#[derive(Debug, Parser)]
enum Cmd {
    Config(ConfigCmd),
    Insert(InsertCmd),
    Queued(QueuedCmd),
    ProcessEvents(ProcessEventsCmd),
}

#[derive(Debug, Parser)]
struct ConfigCmd {
    #[clap(long)]
    output: Option<PathBuf>,
}

impl ConfigCmd {
    fn run(&self, _args: &Args, config: &Config) -> Result<(), CibError> {
        let json = config.to_json();

        if let Some(output) = &self.output {
            write(output, json.as_bytes()).map_err(|e| CibError::write_config(output, e))?;
        } else {
            println!("{json}");
        }

        Ok(())
    }
}

#[derive(Debug, Parser)]
struct InsertCmd {}

impl InsertCmd {
    fn run(&self, args: &Args, config: &Config) -> Result<(), CibError> {
        let adder = QueueAdderBuilder::default()
            .db(args.open_db(config)?)
            .filters(&config.filters)
            .build()
            .map_err(CibError::QueueAdder)?;
        let thread = adder.add_events_in_thread();
        thread
            .join()
            .expect("wait for thread to finish")
            .map_err(CibError::add_events)?;
        debug!("cib ends");
        Ok(())
    }
}

#[derive(Debug, Parser)]
struct QueuedCmd {}

impl QueuedCmd {
    fn run(&self, args: &Args, config: &Config) -> Result<(), CibError> {
        let db = args.open_db(config)?;

        let mut broker = Broker::new(config.db()).map_err(CibError::new_broker)?;
        let spec =
            config
                .adapter(&config.default_adapter)
                .ok_or(CibError::UnknownDefaultAdapter(
                    config.default_adapter.clone(),
                ))?;
        let adapter = Adapter::new(&spec.command)
            .with_environment(spec.envs())
            .with_environment(spec.sensitive_envs());
        debug!("default adapter: {adapter:?}");
        broker.set_default_adapter(&adapter);

        let processor = QueueProcessorBuilder::default()
            .db(db)
            .broker(broker)
            .build()
            .map_err(CibError::process_queue)?;
        let thread = processor.process_in_thread();
        thread
            .join()
            .expect("wait for thread to finish")
            .map_err(CibError::process_queue)?;

        debug!("cib ends");
        Ok(())
    }
}

#[derive(Debug, Parser)]
struct ProcessEventsCmd {}

impl ProcessEventsCmd {
    fn run(&self, args: &Args, config: &Config) -> Result<(), CibError> {
        let adder = QueueAdderBuilder::default()
            .db(args.open_db(config)?)
            .filters(&config.filters)
            .push_shutdown()
            .build()
            .map_err(CibError::QueueAdder)?;
        let adder = adder.add_events_in_thread();

        let profile = Profile::load().map_err(CibError::profile)?;

        let db = args.open_db(config)?;

        let mut page = PageBuilder::default()
            .node_alias(&profile.config.node.alias)
            .runs(db.get_all_runs().map_err(CibError::db)?)
            .build()
            .map_err(CibError::page_updater)?;

        if let Some(dirname) = &config.report_dir {
            page.set_output_dir(dirname);
        }
        page.update_in_thread(db, &profile.config.node.alias, false);

        let mut broker = Broker::new(config.db()).map_err(CibError::new_broker)?;
        let spec =
            config
                .adapter(&config.default_adapter)
                .ok_or(CibError::UnknownDefaultAdapter(
                    config.default_adapter.clone(),
                ))?;
        let adapter = Adapter::new(&spec.command)
            .with_environment(spec.envs())
            .with_environment(spec.sensitive_envs());
        debug!("default adapter: {adapter:?}");
        broker.set_default_adapter(&adapter);

        let processor = QueueProcessorBuilder::default()
            .db(args.open_db(config)?)
            .broker(broker)
            .build()
            .map_err(CibError::process_queue)?;
        let processor = processor.process_in_thread();
        processor
            .join()
            .expect("wait for processor thread to finish")
            .map_err(CibError::process_queue)?;

        adder
            .join()
            .expect("wait for adder thread to finish")
            .map_err(CibError::add_events)?;

        debug!("cib ends");
        Ok(())
    }
}

#[derive(Debug, thiserror::Error)]
enum CibError {
    #[error("failed to read configuration file {0}")]
    ReadConfig(PathBuf, #[source] ConfigError),

    #[error("failed to write config as JSON to file {0}")]
    WriteConfig(PathBuf, #[source] std::io::Error),

    #[error("failed to look up node profile")]
    Profile(#[source] radicle::profile::Error),

    #[error("failed to use SQLite database")]
    Db(#[source] DbError),

    #[error("failed to create report page")]
    PageUpdater(#[source] PageError),

    #[error("failed create broker data type")]
    NewBroker(#[source] BrokerError),

    #[error("failed to add node events into queue")]
    QueueAdder(#[source] AdderError),

    #[error("failed to process events from queue")]
    ProcessQueue(#[source] QueueError),

    #[error("failed to add events to queue")]
    AddEvents(#[source] AdderError),

    #[error("default adapter is not in list of adapters")]
    UnknownDefaultAdapter(String),
}

impl CibError {
    fn read_config(filename: &Path, e: ConfigError) -> Self {
        Self::ReadConfig(filename.into(), e)
    }

    fn write_config(filename: &Path, e: std::io::Error) -> Self {
        Self::WriteConfig(filename.into(), e)
    }

    fn profile(e: radicle::profile::Error) -> Self {
        Self::Profile(e)
    }

    fn db(e: DbError) -> Self {
        Self::Db(e)
    }

    fn page_updater(e: PageError) -> Self {
        Self::PageUpdater(e)
    }

    fn new_broker(e: BrokerError) -> Self {
        Self::NewBroker(e)
    }

    fn process_queue(e: QueueError) -> Self {
        Self::ProcessQueue(e)
    }

    fn add_events(e: AdderError) -> Self {
        Self::AddEvents(e)
    }
}
