//! A management tool for the CI broker.
//!
//! This tool lets the node operator query and manage the CI broker's
//! database: the persistent event queue, the list of CI runs, etc.
//!
//! The tool is also used as part of the CI broker's acceptance test
//! suite (see the `ci-broker.subplot` document).

use std::{
    error::Error,
    fs::{read, write},
    path::PathBuf,
    process::exit,
    str::FromStr,
};

use clap::Parser;

use radicle::{
    git::RefString,
    prelude::{NodeId, RepoId},
    storage::ReadStorage,
    Profile, Storage,
};
use radicle_git_ext::Oid;

use radicle_ci_broker::{
    broker::BrokerError,
    db::{Db, DbError, QueueId},
    event::BrokerEvent,
    msg::{RunId, RunResult},
    pages::{PageBuilder, PageError},
    run::{Run, Whence},
};

fn main() {
    if let Err(e) = fallible_main() {
        eprintln!("ERROR: {}", e);
        let mut e = e.source();
        while let Some(source) = e {
            eprintln!("caused by: {}", source);
            e = source.source();
        }
        exit(1);
    }
}

#[allow(clippy::result_large_err)]
fn fallible_main() -> Result<(), CibToolError> {
    pretty_env_logger::init();

    let args = Args::parse();
    args.run()?;

    Ok(())
}

/// Radicle CI broker management tool for node operators.
///
/// Query and update the CI broker database file: the queue of events
/// waiting to be processed, the list of CI runs. Also, generate HTML
/// report pages from the database.
///
/// This tool can be used whether the CI broker is running or not.
#[derive(Parser)]
struct Args {
    /// Name of the SQLite database file. The file will be created if
    /// it does not already exist. Locking is handled automatically.
    #[clap(long)]
    db: PathBuf,

    #[clap(subcommand)]
    cmd: Cmd,
}

impl Args {
    #[allow(clippy::result_large_err)]
    fn run(&self) -> Result<(), CibToolError> {
        match &self.cmd {
            Cmd::Counter(x) => x.run(self)?,
            Cmd::Event(x) => x.run(self)?,
            Cmd::Run(x) => x.run(self)?,
            Cmd::Report(x) => x.run(self)?,
        }
        Ok(())
    }

    #[allow(clippy::result_large_err)]
    fn open_db(&self) -> Result<Db, CibToolError> {
        Ok(Db::new(&self.db)?)
    }
}

#[derive(Parser)]
enum Cmd {
    /// Manage a counter in the database. This is meant to be used
    /// only by the CI broker test suite, not by people.
    #[clap(hide = true)]
    Counter(CounterCmd),

    /// Manage the event queue. The events are for Git refs having
    /// changed.
    Event(EventCmd),

    /// Manage the list of CI runs.
    Run(RunCmd),

    /// Produce HTML reports based on database contents.
    Report(ReportCmd),
}

#[derive(Parser)]
struct CounterCmd {
    #[clap(subcommand)]
    cmd: CounterSubCmd,
}

impl CounterCmd {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        match &self.cmd {
            CounterSubCmd::Show(x) => x.run(args)?,
            CounterSubCmd::Count(x) => x.run(args)?,
        }
        Ok(())
    }
}

#[derive(Parser)]
enum CounterSubCmd {
    /// Show the current value of the counter.
    Show(ShowCounter),

    /// Count until the counter reaches a minimum value.
    Count(CountCounter),
}

#[derive(Parser)]
struct ShowCounter {}

impl ShowCounter {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let db = args.open_db()?;
        let counter = db.get_counter()?;
        println!("{}", counter.unwrap_or(0));
        Ok(())
    }
}

#[derive(Parser)]
struct CountCounter {
    /// The minimum value which counting aims at.
    #[clap(long)]
    goal: i64,
}

impl CountCounter {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let db = args.open_db()?;
        Self::inc(&db, self.goal)?;

        Ok(())
    }

    fn inc(db: &Db, goal: i64) -> Result<(), DbError> {
        let mut prev: i64 = -1;
        loop {
            db.begin()?;
            println!("BEGIN");

            let current = db.get_counter()?;
            println!("  current as read={current:?}");
            let current = current.unwrap_or(0);
            println!("  current: {current}; prev={prev}");
            if current < prev {
                panic!("current < prev");
            }
            if current >= goal {
                println!("GOAL");
                db.rollback()?;
                println!("ROLLBACK");
                break;
            }

            let new = current + 1;
            if (new == 1 && db.create_counter(new).is_err()) || db.update_counter(new).is_err() {
                db.rollback()?;
                println!("ROLLBACK");
            } else {
                println!("  increment to {new}");
                db.commit()?;
                println!("COMMIT");
                prev = new;
            }
        }

        Ok(())
    }
}

#[derive(Parser)]
struct EventCmd {
    #[clap(subcommand)]
    cmd: EventSubCmd,
}

impl EventCmd {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        match &self.cmd {
            EventSubCmd::Add(x) => x.run(args)?,
            EventSubCmd::Shutdown(x) => x.run(args)?,
            EventSubCmd::List(x) => x.run(args)?,
            EventSubCmd::Count(x) => x.run(args)?,
            EventSubCmd::Show(x) => x.run(args)?,
            EventSubCmd::Remove(x) => x.run(args)?,
        }
        Ok(())
    }
}

#[derive(Parser)]
enum EventSubCmd {
    /// List events in the queue, waiting to be processed.
    List(ListEvents),

    /// Show the number of events in the queue.
    Count(CountEvents),

    /// Add an event to the queue.
    Add(AddEvent),

    /// Add a shutdown event to the queue.
    ///
    /// The shutdown event causes the CI broker to terminate.
    Shutdown(Shutdown),

    /// Show an event in the queue.
    Show(ShowEvent),

    /// Remove an event from the queue.
    Remove(RemoveEvent),
}

#[derive(Parser)]
struct ListEvents {
    /// Show more details about the event.
    #[clap(long)]
    verbose: bool,
}

impl ListEvents {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let db = args.open_db()?;
        for id in db.queued_events()? {
            if self.verbose {
                if let Some(e) = db.get_queued_event(&id)? {
                    println!("{id}: {:?}", e);
                } else {
                    println!("{id}: No such event");
                }
            } else {
                println!("{id}");
            }
        }
        Ok(())
    }
}

#[derive(Parser)]
struct CountEvents {}

impl CountEvents {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let db = args.open_db()?;
        println!("{}", db.queued_events()?.len());
        Ok(())
    }
}

#[derive(Parser)]
struct AddEvent {
    /// Set the repository the event refers to. Can be a RID, or the
    /// repository name.
    #[clap(long)]
    repo: String,

    /// Set the name of the ref the event refers to.
    #[clap(long, alias = "ref")]
    name: String,

    /// Set the commit the event refers to. Can be the SHA1 commit id,
    /// or a symbolic Git revision, as understood by `git rev-parse`.
    /// For example, `HEAD`.
    #[clap(long)]
    commit: String,

    /// The base commit referred to by the event. Optional, but must
    /// be a SHA commit id.
    #[clap(long)]
    base: Option<Oid>,

    /// Write the event to this file, as JSON, instead of adding it to
    /// the queue.
    #[clap(long)]
    output: Option<PathBuf>,

    /// Write the event ID to this file, after adding the event to the
    /// queue.
    #[clap(long)]
    id_file: Option<PathBuf>,
}

impl AddEvent {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let rid = if let Ok(rid) = RepoId::from_urn(&self.repo) {
            rid
        } else {
            self.lookup_rid(&self.repo)?
        };

        let oid = if let Ok(rid) = Oid::from_str(&self.commit) {
            rid
        } else {
            self.lookup_commit(rid, &self.commit)?
        };

        let name = format!(
            "refs/namespaces/{}/refs/heads/{}",
            self.lookup_nid()?,
            self.name.as_str()
        );
        let name = RefString::try_from(name).expect("RefString");

        let event = BrokerEvent::new(&rid, &name, &oid, self.base);

        if let Some(output) = &self.output {
            let json = serde_json::to_string_pretty(&event)
                .map_err(|e| CibToolError::EventToJson(event.clone(), e))?;
            std::fs::write(output, json.as_bytes())
                .map_err(|e| CibToolError::Write(output.into(), e))?;
        } else {
            let db = args.open_db()?;
            let id = db.push_queued_event(event)?;
            println!("{id}");

            if let Some(filename) = &self.id_file {
                write(filename, id.to_string().as_bytes()).expect("write id file");
            }
        }
        Ok(())
    }

    #[allow(clippy::result_large_err)]
    fn lookup_nid(&self) -> Result<NodeId, CibToolError> {
        let profile = Profile::load().map_err(CibToolError::Profile)?;
        Ok(*profile.id())
    }

    #[allow(clippy::result_large_err)]
    fn lookup_rid(&self, wanted: &str) -> Result<RepoId, CibToolError> {
        let profile = Profile::load().map_err(CibToolError::Profile)?;
        let storage =
            Storage::open(profile.storage(), profile.info()).map_err(CibToolError::Storage)?;

        let mut rid = None;
        let repo_infos = storage.repositories().map_err(CibToolError::Repositories)?;
        for ri in repo_infos {
            let project = ri
                .doc
                .project()
                .map_err(|e| CibToolError::Project(ri.rid, e))?;

            if project.name() == wanted {
                if rid.is_some() {
                    return Err(CibToolError::DuplicateRepositories(wanted.into()));
                }
                rid = Some(ri.rid);
            }
        }

        if let Some(rid) = rid {
            Ok(rid)
        } else {
            Err(CibToolError::NotFound(wanted.into()))
        }
    }

    #[allow(clippy::result_large_err)]
    fn lookup_commit(&self, rid: RepoId, gitref: &str) -> Result<Oid, CibToolError> {
        let profile = Profile::load().map_err(CibToolError::Profile)?;
        let storage =
            Storage::open(profile.storage(), profile.info()).map_err(CibToolError::Storage)?;
        let repo = storage
            .repository(rid)
            .map_err(|e| CibToolError::RepoOpen(rid, e))?;
        let object = repo
            .backend
            .revparse_single(gitref)
            .map_err(|e| CibToolError::RevParse(gitref.into(), e))?;

        Ok(object.id().into())
    }
}

#[derive(Parser)]
struct ShowEvent {
    /// ID of event to show.
    #[clap(long, required_unless_present = "id_file")]
    id: Option<QueueId>,

    /// Show event as JSON? Default is a debugging format useful for
    /// programmers.
    #[clap(long)]
    json: bool,

    /// Write output to this file.
    #[clap(long)]
    output: Option<PathBuf>,

    /// Read ID of event to show from this file.
    #[clap(long)]
    id_file: Option<PathBuf>,
}

impl ShowEvent {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let db = args.open_db()?;

        let id = if let Some(id) = &self.id {
            id.clone()
        } else {
            assert!(self.id_file.is_some());
            let file = self.id_file.as_ref().unwrap();
            let id = read(file).expect("read id file");
            let id = String::from_utf8_lossy(&id).to_string();
            QueueId::from(&id)
        };

        if let Some(event) = db.get_queued_event(&id)? {
            if self.json {
                let json = serde_json::to_string_pretty(&event.event())
                    .map_err(|e| CibToolError::EventToJson(event.event().clone(), e))?;
                if let Some(filename) = &self.output {
                    std::fs::write(filename, json.as_bytes())
                        .map_err(|e| CibToolError::Write(filename.into(), e))?;
                } else {
                    println!("{json}");
                }
            } else {
                println!("{event:#?}");
            }
        }
        Ok(())
    }
}

#[derive(Parser)]
struct RemoveEvent {
    /// ID of event to remove.
    #[clap(long, required_unless_present = "id_file")]
    id: Option<QueueId>,

    /// Read ID of event to remove from this file.
    #[clap(long)]
    id_file: Option<PathBuf>,
}

impl RemoveEvent {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let db = args.open_db()?;

        let id = if let Some(id) = &self.id {
            id.clone()
        } else {
            assert!(self.id_file.is_some());
            let file = self.id_file.as_ref().unwrap();
            let id = read(file).expect("read id file");
            let id = String::from_utf8_lossy(&id).to_string();
            QueueId::from(&id)
        };

        db.remove_queued_event(&id)?;
        Ok(())
    }
}

#[derive(Parser)]
struct Shutdown {
    /// Write ID of the event to this file, after adding the event to
    /// the queue.
    #[clap(long)]
    id_file: Option<PathBuf>,
}

impl Shutdown {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let db = args.open_db()?;
        let id = db.push_queued_event(BrokerEvent::Shutdown)?;

        if let Some(filename) = &self.id_file {
            write(filename, id.to_string().as_bytes()).expect("write id file");
        }

        Ok(())
    }
}

#[derive(Parser)]
struct RunCmd {
    #[clap(subcommand)]
    cmd: RunSubCmd,
}

impl RunCmd {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        match &self.cmd {
            RunSubCmd::Add(x) => x.run(args)?,
            RunSubCmd::List(x) => x.run(args)?,
        }
        Ok(())
    }
}

#[derive(Parser)]
enum RunSubCmd {
    /// Add information about a CI run to the database.
    Add(AddRun),

    /// List known CI runs on this node to the database.
    List(ListRuns),
}

#[derive(Parser)]
struct AddRun {
    /// Set the run ID.
    #[clap(long)]
    id: RunId,

    /// Set alias of node that performed the CI run.
    #[clap(long)]
    alias: String,

    /// Set optional URL to information about the CI run.
    #[clap(long)]
    url: Option<String>,

    /// Set the repository ID that the CI run for.
    #[clap(long)]
    repo: RepoId,

    /// Set timestamp of the CI run.
    #[clap(long)]
    timestamp: String,

    /// Set the Git branch used by the CI run.
    #[clap(long)]
    branch: String,

    /// Set the commit SHA ID used by the CI run.
    #[clap(long)]
    commit: Oid,

    /// Set the author of the commit used by the CI run.
    #[clap(long)]
    who: Option<String>,

    /// Set the state of the CI run to "triggered".
    #[clap(long, required_unless_present_any = ["running", "finished"])]
    triggered: bool,

    /// Set the state of the CI run to "running".
    #[clap(long)]
    #[clap(long, required_unless_present_any = ["triggered", "finished"])]
    running: bool,

    /// Set the state of the CI run to "finished".
    #[clap(long)]
    #[clap(long, required_unless_present_any = ["triggered", "running"])]
    finished: bool,

    /// Mark the finished CI run as successful.
    #[clap(long, required_unless_present = "failure")]
    success: bool,
    /// Mark the finished CI run as having failed.

    #[clap(long, required_unless_present = "success")]
    failure: bool,
}

impl AddRun {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let db = args.open_db()?;

        let whence = Whence::Branch {
            name: self.branch.clone(),
            commit: self.commit,
            who: self.who.clone(),
        };
        let mut run = Run::new(self.repo, &self.alias, whence, self.timestamp.clone());
        run.set_adapter_run_id(RunId::default());
        if let Some(url) = &self.url {
            run.set_adapter_info_url(url);
        }

        run.set_result(if self.success {
            RunResult::Success
        } else {
            RunResult::Failure
        });

        db.push_run(run).unwrap();

        Ok(())
    }
}

#[derive(Parser)]
struct ListRuns {}

impl ListRuns {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let db = args.open_db()?;

        for run in db.get_all_runs()? {
            println!("{} {run:#?}", run.adapter_run_id().unwrap());
        }

        Ok(())
    }
}

#[derive(Parser)]
struct ReportCmd {
    /// Write HTML files to this directory. The directory must exist:
    /// it is not created automatically.
    #[clap(long)]
    output_dir: PathBuf,
}

impl ReportCmd {
    #[allow(clippy::result_large_err)]
    fn run(&self, args: &Args) -> Result<(), CibToolError> {
        let profile = Profile::load().map_err(CibToolError::Profile)?;

        let db = args.open_db()?;

        let mut page = PageBuilder::default()
            .node_alias(&profile.config.node.alias)
            .runs(db.get_all_runs()?)
            .build()?;

        page.set_output_dir(&self.output_dir);
        let thread = page.update_in_thread(db, &profile.config.node.alias, true);
        thread.join().unwrap()?;

        Ok(())
    }
}

#[derive(Debug, thiserror::Error)]
#[allow(clippy::large_enum_variant)]
enum CibToolError {
    #[error("failed to look up node profile")]
    Profile(#[source] radicle::profile::Error),

    #[error("failed to look up open node storage")]
    Storage(#[source] radicle::storage::Error),

    #[error("failed to list repositories in node storage")]
    Repositories(#[source] radicle::storage::Error),

    #[error("failed to look up project info for repository {0}")]
    Project(RepoId, #[source] radicle::identity::doc::PayloadError),

    #[error("node has more than one repository called {0}")]
    DuplicateRepositories(String),

    #[error("node has no repository called: {0}")]
    NotFound(String),

    #[error("failed to open git repository in node storage: {0}")]
    RepoOpen(RepoId, #[source] radicle::storage::RepositoryError),

    #[error("failed to parse git ref as a commit id: {0}")]
    RevParse(String, #[source] radicle::git::raw::Error),

    #[error(transparent)]
    Broker(#[from] BrokerError),

    #[error(transparent)]
    Db(#[from] DbError),

    #[error("failed to serialize broker event to JSON: {0:#?}")]
    EventToJson(BrokerEvent, #[source] serde_json::Error),

    #[error("failed to write file: {0}")]
    Write(PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Page(#[from] PageError),
}
