//! Read node events, expressed as JSON, from files, and output them as
//! single-line JSON objects to a Unix domain socket.
//!
//! This is a helper tool for testing of the CI broker as a whole.
//!
//! This program forks and runs itself in the background.

use std::{
    env::current_exe,
    fs::{read, remove_file, File, OpenOptions},
    io::{Read, Write},
    net::Shutdown,
    os::unix::net::UnixListener,
    path::PathBuf,
    process::{Command, Stdio},
    thread::sleep,
    time::Duration,
};

use clap::Parser;

use radicle::node::Event;

fn log(file: &mut File, msg: String) {
    file.write_all(msg.as_bytes()).ok();
}

fn main() -> anyhow::Result<()> {
    let args = Args::parse();
    let exe = current_exe()?;

    if args.daemon {
        let mut f = OpenOptions::new()
            .create(true)
            .append(true)
            .open(args.log.unwrap())?;

        log(&mut f, "daemon starts\n".into());

        let repeat = args.repeat.unwrap_or(1);
        log(&mut f, format!("repeat={repeat}\n"));

        let mut events = vec![];
        for filename in args.events.iter() {
            log(
                &mut f,
                format!("daemon reads file {}\n", filename.display()),
            );

            let data = read(filename)?;
            let e: Event = serde_json::from_slice(&data)?;
            let mut e = serde_json::to_string(&e)?;
            e.push('\n');
            events.push(e);
        }

        log(
            &mut f,
            format!("daemon connects to {}\n", args.socket.display()),
        );
        let listener = UnixListener::bind(&args.socket)?;

        if let Some(stream) = listener.incoming().next() {
            log(&mut f, "daemon accepted connection\n".into());
            let mut stream = stream?;
            for e in events.iter() {
                for _ in 0..repeat {
                    stream.write_all(e.as_bytes())?;
                }
                log(&mut f, "daemon sent a message\n".into());
            }
            log(&mut f, "daemon sent all messages\n".into());

            stream.shutdown(Shutdown::Write)?;
            log(&mut f, "daemon shutdown write\n".into());

            stream.set_read_timeout(Some(Duration::from_millis(1000)))?;
            let mut buf = vec![];
            stream.read_to_end(&mut buf).ok();
            log(&mut f, "daemon finished reading\n".into());
        }

        remove_file(&args.socket)?;
        log(&mut f, "daemon ends\n".into());
    } else {
        if args.socket.exists() {
            eprintln!("removing socket {}", args.socket.display());
            remove_file(&args.socket)?;
        }
        eprintln!("launching daemon");
        let repeat = args.repeat.unwrap_or(1);
        Command::new(exe)
            .arg("--daemon")
            .arg("--log")
            .arg(args.log.unwrap_or(PathBuf::from("/dev/null")))
            .arg(&format!("--repeat={repeat}"))
            .arg(&args.socket)
            .args(&args.events)
            .stdin(Stdio::null())
            .stdout(Stdio::null())
            .stderr(Stdio::null())
            .spawn()?;
        eprintln!("waiting for daemon to create socket");
        while !args.socket.exists() {
            eprintln!("no socket yet");
            sleep(Duration::from_millis(100));
        }
        eprintln!("there is a socket now");
    }

    Ok(())
}

#[derive(Debug, Parser)]
struct Args {
    #[clap(long)]
    daemon: bool,
    #[clap(long)]
    log: Option<PathBuf>,
    #[clap(long)]
    repeat: Option<usize>,
    socket: PathBuf,
    events: Vec<PathBuf>,
}
