use std::thread::{spawn, JoinHandle};

use radicle::Profile;

use log::{debug, info};

use crate::{
    db::{Db, DbError},
    event::{BrokerEvent, EventFilter, NodeEventError, NodeEventSource},
};

#[derive(Default)]
pub struct QueueAdderBuilder {
    db: Option<Db>,
    filters: Option<Vec<EventFilter>>,
    push_shutdown: bool,
}

impl QueueAdderBuilder {
    pub fn build(self) -> Result<QueueAdder, AdderError> {
        Ok(QueueAdder {
            db: self.db.ok_or(AdderError::Missing("db"))?,
            filters: self.filters.ok_or(AdderError::Missing("filters"))?,
            push_shutdown: self.push_shutdown,
        })
    }

    pub fn db(mut self, db: Db) -> Self {
        self.db = Some(db);
        self
    }

    pub fn filters(mut self, filters: &[EventFilter]) -> Self {
        self.filters = Some(filters.to_vec());
        self
    }

    pub fn push_shutdown(mut self) -> Self {
        self.push_shutdown = true;
        self
    }
}

pub struct QueueAdder {
    filters: Vec<EventFilter>,
    db: Db,
    push_shutdown: bool,
}

impl QueueAdder {
    pub fn add_events_in_thread(self) -> JoinHandle<Result<(), AdderError>> {
        spawn(move || self.add_events())
    }

    pub fn add_events(&self) -> Result<(), AdderError> {
        let profile = Profile::load()?;
        debug!("loaded profile {profile:#?}");

        let mut source = NodeEventSource::new(&profile)?;
        debug!("created node event source");

        for filter in self.filters.iter() {
            source.allow(filter.clone());
        }
        debug!("added filters to node event source");

        // This loop ends when there's an error, e.g., failure to read an
        // event from the node.
        'event_loop: loop {
            debug!("waiting for event from node");
            let events = source.event()?;
            if events.is_empty() {
                info!("no more events from node control sockets");
                break 'event_loop;
            } else {
                for e in events {
                    debug!("got event {e:#?}");
                    info!("insert broker event into queue: {e:?}");
                    self.db.push_queued_event(e)?;
                }
            }
        }

        // Add a shutdown event to the queue so that the queue
        // processing thread knows to stop.
        if self.push_shutdown {
            info!("push a shutdown event into queue");
            self.db.push_queued_event(BrokerEvent::Shutdown)?;
        }

        Ok(())
    }
}

#[derive(Debug, thiserror::Error)]
pub enum AdderError {
    #[error("programming error: QueueAdderBuilder field {0} was not set")]
    Missing(&'static str),

    #[error(transparent)]
    Profile(#[from] radicle::profile::Error),

    #[error(transparent)]
    NodeEvent(#[from] NodeEventError),

    #[error(transparent)]
    Db(#[from] DbError),
}
