//! Process events in the persistent event queue.

#![allow(clippy::result_large_err)]

use std::{
    collections::HashSet,
    sync::{
        mpsc::{sync_channel, Receiver, RecvTimeoutError, SyncSender},
        Arc, Mutex,
    },
    thread::JoinHandle,
    time::{Duration, Instant},
};

use radicle::{identity::RepoId, Profile};

use crate::{
    adapter::{Adapter, Adapters},
    broker::{Broker, BrokerError},
    ci_event::{CiEvent, CiEventV1},
    db::{Db, DbError, QueuedCiEvent},
    filter::{EventFilter, Trigger},
    logger,
    msg::{MessageError, RequestBuilder},
    notif::{NotificationReceiver, NotificationSender},
    pull_queue::{PullQueue, PullQueueError},
    worker::{start_thread, Worker},
};

#[derive(Default)]
pub struct QueueProcessorBuilder {
    db: Option<Db>,
    broker: Option<Broker>,
    filters: Option<Vec<EventFilter>>,
    triggers: Option<Vec<Trigger>>,
    adapters: Option<Adapters>,
    events_rx: Option<NotificationReceiver>,
    run_tx: Option<NotificationSender>,
    queue_len_interval: Option<Duration>,
    concurrent_adapters: Option<usize>,
}

const DEFAULT_QUEUE_LEN_DURATION: Duration = Duration::from_secs(10);

impl QueueProcessorBuilder {
    pub fn build(self) -> Result<QueueProcessor, QueueError> {
        let profile = Profile::load().map_err(QueueError::Profile)?;
        let broker = self.broker.ok_or(QueueError::Missing("broker"))?;
        let filters = self.filters.ok_or(QueueError::Missing("filters"))?;
        let triggers = self.triggers.ok_or(QueueError::Missing("triggers"))?;
        let adapters = self.adapters.ok_or(QueueError::Missing("adapters"))?;
        let run_tx = self.run_tx.ok_or(QueueError::Missing("run_tx"))?;
        let concurrent_adapters = self
            .concurrent_adapters
            .ok_or(QueueError::Missing("concurrent_adapters"))?;
        let (procssed_tx, processed_rx) = sync_channel(1);

        let picked_events = PullQueue::new();

        let mut processors = vec![];
        for _ in 0..concurrent_adapters {
            let broker = Broker::new(broker.db().filename(), broker.max_run_time())
                .map_err(QueueError::NewBroker)?;
            let event_procssor = EventProcessor::new(
                profile.clone(),
                filters.clone(),
                triggers.clone(),
                adapters.clone(),
                broker,
                run_tx.clone(),
                picked_events.clone(),
                procssed_tx.clone(),
            );
            processors.push(start_thread(event_procssor));
        }

        Ok(QueueProcessor {
            db: self.db.ok_or(QueueError::Missing("db"))?,
            events_rx: self.events_rx.ok_or(QueueError::Missing("events_rx"))?,
            queue_len_interval: self
                .queue_len_interval
                .unwrap_or(DEFAULT_QUEUE_LEN_DURATION),
            prev_queue_len: Instant::now(),
            processors,
            processed_rx: Some(processed_rx),
            picked_events: Some(picked_events),
            current: CurrentlyPicked::default(),
        })
    }

    pub fn events_rx(mut self, rx: NotificationReceiver) -> Self {
        self.events_rx = Some(rx);
        self
    }

    pub fn run_tx(mut self, tx: NotificationSender) -> Self {
        self.run_tx = Some(tx);
        self
    }

    pub fn db(mut self, db: Db) -> Self {
        self.db = Some(db);
        self
    }

    pub fn queue_len_interval(mut self, interval: Duration) -> Self {
        self.queue_len_interval = Some(interval);
        self
    }

    pub fn concurrent_adapters(mut self, n: usize) -> Self {
        self.concurrent_adapters = Some(n);
        self
    }

    pub fn broker(mut self, broker: Broker) -> Self {
        self.broker = Some(broker);
        self
    }

    pub fn filters(mut self, filters: &[EventFilter]) -> Self {
        self.filters = Some(filters.to_vec());
        self
    }

    pub fn triggers(mut self, triggers: &[Trigger]) -> Self {
        self.triggers = Some(triggers.to_vec());
        self
    }

    pub fn adapters(mut self, adapters: &Adapters) -> Self {
        self.adapters = Some(adapters.clone());
        self
    }
}

impl Worker for QueueProcessor {
    const NAME: &str = "queue-processor";
    type Error = QueueError;
    fn work(&mut self) -> Result<(), QueueError> {
        logger::queueproc_start(self.processors.len());

        // Spawn a thread to process results from running adapters.
        #[allow(clippy::unwrap_used)]
        let adapter_results = start_thread(AdapterResults::new(
            self.processed_rx.take().unwrap(),
            self.current.clone(),
        ));

        // Pick events from queue, send to worker threads that run
        // adapters. Results are processed by thread above.
        let result = self.process_until_shutdown();

        logger::queueproc_end(&result);

        // Wait for worker threads to terminate. This closes all
        // sender ends for results channel.
        while let Some(proc) = self.processors.pop() {
            let result = proc
                .join()
                .map_err(|_| QueueError::JoinEventProcessorThread)?;
            logger::queueproc_processor_thread_result(result.as_ref().map(|_| ()));
        }

        // Wait for results processing thread to terminate.
        adapter_results.join().ok();
        Ok(())
    }
}

pub struct QueueProcessor {
    db: Db,
    events_rx: NotificationReceiver,
    queue_len_interval: Duration,
    prev_queue_len: Instant,
    picked_events: Option<PullQueue<Picked>>,
    processors: Vec<JoinHandle<Result<(), QueueError>>>,
    processed_rx: Option<Receiver<RepoId>>,
    current: CurrentlyPicked,
}

impl QueueProcessor {
    #[allow(clippy::unwrap_used)]
    fn process_until_shutdown(&mut self) -> Result<(), QueueError> {
        let mut done = false;
        let mut picked_events = self.picked_events.take().unwrap();
        while !done {
            // Process all events currently in the event queue, if a
            // filter allows it.
            loop {
                if let Some(qe) = self.pick_event()? {
                    let picked = Picked::new(qe.clone());
                    self.current.insert(picked.qe.event().repository());
                    picked_events.push(picked).map_err(QueueError::PullQueue)?;

                    // We always drop the picked event as soon as it has
                    // been pushed to a processing thread, so that if
                    // anything goes wrong, we're less likely try to
                    // process the same event again. It's better for the
                    // failure to be logged and let the humans deal with
                    // whatever complicated failure situation is
                    // happening.
                    self.drop_event(&qe)?;
                } else if self.current.is_empty() {
                    break;
                }
            }

            // Wait for a notification of new events in the queue.
            // This prevents the loop from being a busy loop.
            match self.events_rx.wait_for_notification() {
                Ok(_) => {}
                Err(RecvTimeoutError::Timeout) => {}
                Err(RecvTimeoutError::Disconnected) => {
                    logger::queueproc_channel_disconnect();
                    done = true;
                }
            }
        }

        Ok(())
    }

    fn pick_event(&mut self) -> Result<Option<QueuedCiEvent>, QueueError> {
        let ids = self.db.queued_ci_events().map_err(QueueError::db)?;

        let elapsed = self.prev_queue_len.elapsed();
        if elapsed > self.queue_len_interval {
            logger::queueproc_queue_length(ids.len());
            self.prev_queue_len = Instant::now();
        }

        let mut queue = vec![];
        for id in ids.iter() {
            if let Some(qe) = self.db.get_queued_ci_event(id).map_err(QueueError::db)? {
                queue.push(qe);
            }
        }
        queue.sort_by_cached_key(|qe| qe.timestamp().to_string());

        // If we can access the set of repositories for which CI is
        // currently running, remove those repositories from the list.
        let ids = self.current.list();
        queue = queue
            .iter()
            .filter(|qe| {
                if let Some(repoid) = qe.event().repository() {
                    !ids.contains(repoid)
                } else {
                    true
                }
            })
            .cloned()
            .collect();

        if let Some(qe) = queue.first() {
            Ok(Some(qe.clone()))
        } else {
            Ok(None)
        }
    }

    fn drop_event(&mut self, qe: &QueuedCiEvent) -> Result<(), QueueError> {
        logger::queueproc_remove_event(qe);
        self.db
            .remove_queued_ci_event(qe.id())
            .map_err(QueueError::db)?;
        Ok(())
    }
}

#[derive(Default, Clone)]
struct CurrentlyPicked {
    set: Arc<Mutex<HashSet<RepoId>>>,
}

impl CurrentlyPicked {
    fn insert(&mut self, repoid: Option<&RepoId>) {
        if let Some(repoid) = repoid {
            if let Ok(mut set) = self.set.lock() {
                set.insert(*repoid);
            }
        }
    }

    fn remove(&mut self, repoid: RepoId) {
        if let Ok(mut set) = self.set.lock() {
            set.remove(&repoid);
        }
    }

    fn list(&self) -> Vec<RepoId> {
        if let Ok(set) = self.set.lock() {
            set.iter().copied().collect()
        } else {
            vec![]
        }
    }

    fn is_empty(&self) -> bool {
        if let Ok(set) = self.set.lock() {
            set.is_empty()
        } else {
            false
        }
    }
}

struct EventProcessor {
    profile: Profile,
    filters: Vec<EventFilter>,
    triggers: Vec<Trigger>,
    adapters: Adapters,
    broker: Broker,
    run_tx: NotificationSender,
    picked_events: PullQueue<Picked>,
    processed_tx: SyncSender<RepoId>,
}

impl EventProcessor {
    #[allow(clippy::too_many_arguments)]
    fn new(
        profile: Profile,
        filters: Vec<EventFilter>,
        triggers: Vec<Trigger>,
        adapters: Adapters,
        broker: Broker,
        run_tx: NotificationSender,
        picked_events: PullQueue<Picked>,
        processed_tx: SyncSender<RepoId>,
    ) -> Self {
        Self {
            profile,
            filters,
            triggers,
            adapters,
            broker,
            run_tx,
            picked_events,
            processed_tx,
        }
    }

    fn process_picked_event(&self, picked: Picked) {
        let res = self.matching_adapters(picked.qe.event());
        logger::queueproc_worker_thread_result(res.as_ref().map(|_| ()));
        if let Ok(Some(adapters)) = res {
            for adapter in adapters {
                self.run_adapter(&picked.qe, &adapter).ok();
            }
        }

        if let Some(repoid) = picked.qe.event().repository() {
            self.processed_tx.send(*repoid).ok();
        }
    }

    fn matching_adapters(&self, e: &CiEvent) -> Result<Option<Vec<Adapter>>, QueueError> {
        let mut adapters = vec![];

        if self.filters.iter().any(|filter| filter.allows(e)) {
            if let Some(default) = self.adapters.default_adapter() {
                adapters.push(default.clone());
            } else {
                return Err(QueueError::NoDefaultAdapter);
            }
        }

        for trigger in self.triggers.iter() {
            if trigger.allows(e) {
                let name = trigger.adapter().to_string();
                let adapter = self
                    .adapters
                    .get(&name)
                    .ok_or(QueueError::UnknownAdapter(name))?;
                adapters.push(adapter.clone());
            }
        }

        if adapters.is_empty() {
            Ok(None)
        } else {
            Ok(Some(adapters))
        }
    }

    fn run_adapter(&self, qe: &QueuedCiEvent, adapter: &Adapter) -> Result<bool, QueueError> {
        let mut done = false;

        self.run_tx.notify()?;
        logger::queueproc_picked_event(qe.id(), qe, adapter);
        let res = self.process_event(qe.event(), adapter);
        logger::queueproc_processed_event(&res);
        match res {
            Ok(shut_down) => done = shut_down == MaybeShutdown::Shutdown,
            Err(QueueError::BuildTrigger(_, _)) => done = false,
            Err(err) => Err(err)?,
        }
        self.run_tx.notify()?;
        Ok(done)
    }

    fn process_event(
        &self,
        event: &CiEvent,
        adapter: &Adapter,
    ) -> Result<MaybeShutdown, QueueError> {
        if matches!(event, CiEvent::V1(CiEventV1::Shutdown)) {
            logger::queueproc_action_shutdown();
            Ok(MaybeShutdown::Shutdown)
        } else {
            logger::queueproc_action_run(event);

            let trigger = RequestBuilder::default()
                .profile(&self.profile)
                .ci_event(event)
                .build_trigger_from_ci_event()
                .map_err(|e| QueueError::build_trigger(event, e));
            logger::queueproc_trigger(&trigger);
            let trigger = trigger?;

            self.broker
                .execute_ci(adapter, &trigger, &self.run_tx)
                .map_err(QueueError::execute_ci)?;

            Ok(MaybeShutdown::Continue)
        }
    }
}

impl Worker for EventProcessor {
    const NAME: &str = "event-processor";
    type Error = QueueError;
    fn work(&mut self) -> Result<(), QueueError> {
        while let Ok(Some(picked)) = self.picked_events.pop() {
            self.process_picked_event(picked);
        }
        Ok(())
    }
}

#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum MaybeShutdown {
    Shutdown,
    Continue,
}

struct AdapterResults {
    rx: Receiver<RepoId>,
    current: CurrentlyPicked,
}

impl AdapterResults {
    fn new(rx: Receiver<RepoId>, current: CurrentlyPicked) -> Self {
        Self { rx, current }
    }
}

impl Worker for AdapterResults {
    const NAME: &str = "adapter-result-processor";
    type Error = QueueError;
    fn work(&mut self) -> Result<(), QueueError> {
        logger::queueproc_start_result_receiver();
        while let Ok(repoid) = self.rx.recv() {
            self.current.remove(repoid);
            logger::queueproc_end_result_receiver();
        }
        Ok(())
    }
}

#[derive(Debug, Clone)]
struct Picked {
    qe: QueuedCiEvent,
}

impl Picked {
    fn new(qe: QueuedCiEvent) -> Self {
        Self { qe }
    }
}

#[derive(Debug, thiserror::Error)]
pub enum QueueError {
    #[error("failed to load node profile")]
    Profile(#[source] radicle::profile::Error),

    #[error("programming error: QueueProcessorBuilder field {0} was not set")]
    Missing(&'static str),

    #[error("failed to use SQLite database")]
    Db(#[source] DbError),

    #[error("failed to create a trigger message from broker event {0:?}")]
    BuildTrigger(CiEvent, #[source] MessageError),

    #[error("failed to run CI")]
    ExecuteCi(#[source] BrokerError),

    #[error(transparent)]
    NotifyRun(#[from] crate::notif::NotificationError),

    #[error("trigger refers to unknown adapter {0}")]
    UnknownAdapter(String),

    #[error("no default adapter specified in configuration")]
    NoDefaultAdapter,

    #[error("failed to send to channel for picked events")]
    SendPicked,

    #[error("failed to receive from channel for picked events")]
    RecvPicked,

    #[error("failed to send to channel for results of processed events")]
    SendProcessResult,

    #[error("failed to receive from channel for results of processed events")]
    RecvProcessResult,

    #[error("failed to wait for thread to process events to finish")]
    JoinEventProcessorThread,

    #[error("failed to wait for thread to run adapters to finish")]
    JoinAdapterThread,

    #[error("failed to wait for thread to process results from adapters to finish")]
    JoinResultThread,

    #[error("failed to create a new broker instance")]
    NewBroker(#[source] BrokerError),

    #[error("failure when using a pull queue")]
    PullQueue(#[source] PullQueueError),
}

impl QueueError {
    fn db(e: DbError) -> Self {
        Self::Db(e)
    }

    fn build_trigger(event: &CiEvent, err: MessageError) -> Self {
        Self::BuildTrigger(event.clone(), err)
    }

    fn execute_ci(e: BrokerError) -> Self {
        Self::ExecuteCi(e)
    }
}
