//! Process events in the persistent event queue.

#![allow(clippy::result_large_err)]

use std::{
    collections::HashSet,
    sync::{Arc, Mutex},
    thread::spawn,
    time::{Duration, Instant},
};

use radicle::{Profile, identity::RepoId};

use crate::{
    adapter::{Adapter, Adapters},
    broker::{Broker, BrokerError},
    ci_event::{CiEvent, CiEventV1},
    db::{Db, DbError, QueuedCiEvent},
    filter::{EventFilter, Trigger},
    logger,
    msg::{MessageError, RequestBuilder},
    notif::{NotificationReceiver, NotificationSender},
    worker::Worker,
};

#[derive(Default)]
pub struct QueueProcessorBuilder {
    db: Option<Db>,
    broker: Option<Broker>,
    filters: Option<Vec<EventFilter>>,
    triggers: Option<Vec<Trigger>>,
    adapters: Option<Adapters>,
    events_rx: Option<NotificationReceiver>,
    run_tx: Option<NotificationSender>,
    queue_len_interval: Option<Duration>,
    concurrent_adapters: Option<usize>,
}

const DEFAULT_QUEUE_LEN_DURATION: Duration = Duration::from_secs(10);

impl QueueProcessorBuilder {
    pub fn build(self) -> Result<QueueProcessor, QueueError> {
        let profile = Profile::load().map_err(QueueError::Profile)?;
        let broker = self.broker.ok_or(QueueError::Missing("broker"))?;
        let filters = self.filters.ok_or(QueueError::Missing("filters"))?;
        let triggers = self.triggers.ok_or(QueueError::Missing("triggers"))?;
        let adapters = self.adapters.ok_or(QueueError::Missing("adapters"))?;
        let run_tx = self.run_tx.ok_or(QueueError::Missing("run_tx"))?;
        let concurrent_adapters = self
            .concurrent_adapters
            .ok_or(QueueError::Missing("concurrent_adapters"))?;

        Ok(QueueProcessor {
            profile,
            broker,
            filters,
            triggers,
            adapters,
            db: self.db.ok_or(QueueError::Missing("db"))?,
            events_rx: self.events_rx.ok_or(QueueError::Missing("events_rx"))?,
            queue_len_interval: self
                .queue_len_interval
                .unwrap_or(DEFAULT_QUEUE_LEN_DURATION),
            prev_queue_len: Instant::now(),
            concurrent_adapters,
            run_tx,
            current: CurrentlyPicked::default(),
        })
    }

    pub fn events_rx(mut self, rx: NotificationReceiver) -> Self {
        self.events_rx = Some(rx);
        self
    }

    pub fn run_tx(mut self, tx: NotificationSender) -> Self {
        self.run_tx = Some(tx);
        self
    }

    pub fn db(mut self, db: Db) -> Self {
        self.db = Some(db);
        self
    }

    pub fn queue_len_interval(mut self, interval: Duration) -> Self {
        self.queue_len_interval = Some(interval);
        self
    }

    pub fn concurrent_adapters(mut self, n: usize) -> Self {
        self.concurrent_adapters = Some(n);
        self
    }

    pub fn broker(mut self, broker: Broker) -> Self {
        self.broker = Some(broker);
        self
    }

    pub fn filters(mut self, filters: &[EventFilter]) -> Self {
        self.filters = Some(filters.to_vec());
        self
    }

    pub fn triggers(mut self, triggers: &[Trigger]) -> Self {
        self.triggers = Some(triggers.to_vec());
        self
    }

    pub fn adapters(mut self, adapters: &Adapters) -> Self {
        self.adapters = Some(adapters.clone());
        self
    }
}

// The queue processor gets events from the event queue in
// the database, and processes them concurrently by running
// the appropriate adapters. To avoid busy looping, the
// `events_rx` channel is used to receive notification that
// an event has been added to the database. Processing will
// end when the channel is closed and the queue is empty,
// or a "shutdown" event is encountered. In case of shutdown,
// any currently running adapters will be allowed to finish.
pub struct QueueProcessor {
    profile: Profile,
    db: Db,
    broker: Broker,
    filters: Vec<EventFilter>,
    triggers: Vec<Trigger>,
    adapters: Adapters,
    concurrent_adapters: usize,
    events_rx: NotificationReceiver,
    queue_len_interval: Duration,
    prev_queue_len: Instant,
    run_tx: NotificationSender,
    current: CurrentlyPicked,
}

impl QueueProcessor {
    fn process_until_shutdown(&mut self) -> Result<(), QueueError> {
        let mut expecting_new_events = true;
        let mut handles = vec![];

        loop {
            let mut queue_was_emptied = false;

            // If we may spawn another adapter, pick an event from the queue
            // and run the adapters.
            if expecting_new_events && handles.len() < self.concurrent_adapters {
                match self.pick_event() {
                    Ok(Some(qe)) => {
                        // Remove picked event from queue so we doh't re-process it. If we don't
                        // do this, and we crash when processing the event, we'll re-process it
                        // again and again. It seems better to discard an event, and skip running
                        // CI, rather then getting stuck on a specific event.
                        self.drop_event(&qe)?;

                        match self.matching_adapters(qe.event()) {
                            Ok(Some(adapters)) => {
                                let p = self.processor()?;
                                let repoid = qe.event().repository().copied();
                                self.current.insert(qe.event().repository());
                                let h = spawn(move || p.pick_and_process_one(qe, adapters));
                                handles.push((repoid, h));
                            }
                            Ok(None) => (),
                            Err(_) => (),
                        }
                    }
                    Ok(None) => queue_was_emptied = true,
                    Err(_) => (),
                }
            }

            // Wait for any threads processing events that have finished. Remove
            // them from the list of currently running threads.
            let mut h2 = vec![];
            for (repoid, h) in handles {
                if h.is_finished() {
                    if let Some(repoid) = repoid {
                        self.current.remove(repoid);
                    }
                    match h.join() {
                        Err(err) => eprintln!("thread join error {err:?}"),
                        Ok(Err(_)) => (),
                        Ok(Ok(MaybeShutdown::Shutdown)) => {
                            expecting_new_events = false;
                            queue_was_emptied = true;
                        }
                        Ok(Ok(MaybeShutdown::Continue)) => (),
                    }
                } else {
                    h2.push((repoid, h));
                }
            }
            handles = h2;

            // If we didn't empty the event queue, but we're still
            // expecting new events, wait for a new event. This prevents
            // a busy loop.
            if expecting_new_events && queue_was_emptied {
                match self.events_rx.wait_for_notification() {
                    Err(_) => {
                        expecting_new_events = false;
                    }
                    Ok(_) => queue_was_emptied = false,
                }
            }

            if handles.is_empty() && !expecting_new_events && queue_was_emptied {
                break;
            }
        }

        Ok(())
    }

    fn processor(&self) -> Result<Processor, QueueError> {
        Ok(Processor {
            profile: self.profile.clone(),
            broker: Broker::new(self.db.filename(), self.broker.max_run_time())
                .map_err(QueueError::NewBroker)?,
            run_tx: self.run_tx.clone(),
        })
    }

    fn pick_event(&mut self) -> Result<Option<QueuedCiEvent>, QueueError> {
        let ids = self.db.queued_ci_events().map_err(QueueError::db)?;

        let elapsed = self.prev_queue_len.elapsed();
        if elapsed > self.queue_len_interval {
            logger::queueproc_queue_length(ids.len());
            self.prev_queue_len = Instant::now();
        }

        let mut queue = vec![];
        for id in ids.iter() {
            if let Some(qe) = self.db.get_queued_ci_event(id).map_err(QueueError::db)? {
                queue.push(qe);
            }
        }
        queue.sort_by_cached_key(|qe| qe.timestamp().to_string());

        // Remove the repositories for which CI is currently running.
        let ids = self.current.list();
        queue = queue
            .iter()
            .filter(|qe| {
                if let Some(repoid) = qe.event().repository() {
                    !ids.contains(repoid)
                } else {
                    true
                }
            })
            .cloned()
            .collect();

        if let Some(qe) = queue.first() {
            eprintln!("picked event: {qe:?}");
            Ok(Some(qe.clone()))
        } else {
            Ok(None)
        }
    }

    fn drop_event(&mut self, qe: &QueuedCiEvent) -> Result<(), QueueError> {
        logger::queueproc_remove_event(qe);
        self.db
            .remove_queued_ci_event(qe.id())
            .map_err(QueueError::db)?;
        Ok(())
    }

    fn matching_adapters(&self, e: &CiEvent) -> Result<Option<Vec<Adapter>>, QueueError> {
        let mut adapters = vec![];

        if self.filters.iter().any(|filter| filter.allows(e)) {
            if let Some(default) = self.adapters.default_adapter() {
                adapters.push(default.clone());
            } else {
                return Err(QueueError::NoDefaultAdapter);
            }
        }

        for trigger in self.triggers.iter() {
            if trigger.allows(e) {
                let name = trigger.adapter().to_string();
                let adapter = self
                    .adapters
                    .get(&name)
                    .ok_or(QueueError::UnknownAdapter(name))?;
                adapters.push(adapter.clone());
            }
        }

        if adapters.is_empty() {
            Ok(None)
        } else {
            Ok(Some(adapters))
        }
    }
}

impl Worker for QueueProcessor {
    const NAME: &str = "queue-processor";
    type Error = QueueError;
    fn work(&mut self) -> Result<(), QueueError> {
        // Pick events from queue, send to worker threads that run
        // adapters. Results are processed by thread above.
        let result = self.process_until_shutdown();

        logger::queueproc_end(&result);

        Ok(())
    }
}

struct Processor {
    profile: Profile,
    broker: Broker,
    run_tx: NotificationSender,
}

impl Processor {
    fn pick_and_process_one(
        &self,
        qe: QueuedCiEvent,
        adapters: Vec<Adapter>,
    ) -> Result<MaybeShutdown, QueueError> {
        for adapter in adapters.iter() {
            self.run_tx.notify()?;
            logger::queueproc_picked_event(qe.id(), &qe, adapter);
            match qe.event() {
                CiEvent::V1(CiEventV1::Shutdown) => {
                    logger::queueproc_action_shutdown();
                    return Ok(MaybeShutdown::Shutdown);
                }
                _ => {
                    logger::queueproc_action_run(qe.event());

                    let trigger = RequestBuilder::default()
                        .profile(&self.profile)
                        .ci_event(qe.event())
                        .build_trigger_from_ci_event()
                        .map_err(|e| QueueError::build_trigger(qe.event(), e));
                    logger::queueproc_trigger(&trigger);
                    let trigger = trigger?;

                    self.broker
                        .execute_ci(adapter, &trigger, &self.run_tx)
                        .map_err(QueueError::execute_ci)?;
                }
            }
        }

        Ok(MaybeShutdown::Continue)
    }
}

#[derive(Default, Clone)]
struct CurrentlyPicked {
    set: Arc<Mutex<HashSet<RepoId>>>,
}

impl CurrentlyPicked {
    fn insert(&mut self, repoid: Option<&RepoId>) {
        if let Some(repoid) = repoid {
            if let Ok(mut set) = self.set.lock() {
                set.insert(*repoid);
            }
        }
    }

    fn remove(&mut self, repoid: RepoId) {
        if let Ok(mut set) = self.set.lock() {
            set.remove(&repoid);
        }
    }

    fn list(&self) -> Vec<RepoId> {
        if let Ok(set) = self.set.lock() {
            set.iter().copied().collect()
        } else {
            vec![]
        }
    }
}

#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum MaybeShutdown {
    Shutdown,
    Continue,
}

#[derive(Debug, thiserror::Error)]
pub enum QueueError {
    #[error("failed to load node profile")]
    Profile(#[source] radicle::profile::Error),

    #[error("failed to open database")]
    OpenDb(#[source] crate::db::DbError),

    #[error("programming error: QueueProcessorBuilder field {0} was not set")]
    Missing(&'static str),

    #[error("failed to use SQLite database")]
    Db(#[source] DbError),

    #[error("failed to create a trigger message from broker event {0:?}")]
    BuildTrigger(CiEvent, #[source] MessageError),

    #[error("failed to run CI")]
    ExecuteCi(#[source] BrokerError),

    #[error(transparent)]
    NotifyRun(#[from] crate::notif::NotificationError),

    #[error("trigger refers to unknown adapter {0}")]
    UnknownAdapter(String),

    #[error("no default adapter specified in configuration")]
    NoDefaultAdapter,

    #[error("failed to send to channel for picked events")]
    SendPicked,

    #[error("failed to receive from channel for picked events")]
    RecvPicked,

    #[error("failed to send to channel for results of processed events")]
    SendProcessResult,

    #[error("failed to receive from channel for results of processed events")]
    RecvProcessResult,

    #[error("failed to wait for thread to process events to finish")]
    JoinEventProcessorThread,

    #[error("failed to wait for thread to run adapters to finish")]
    JoinAdapterThread,

    #[error("failed to wait for thread to process results from adapters to finish")]
    JoinResultThread,

    #[error("failed to create a new broker instance")]
    NewBroker(#[source] BrokerError),
}

impl QueueError {
    fn db(e: DbError) -> Self {
        Self::Db(e)
    }

    fn build_trigger(event: &CiEvent, err: MessageError) -> Self {
        Self::BuildTrigger(event.clone(), err)
    }

    fn execute_ci(e: BrokerError) -> Self {
        Self::ExecuteCi(e)
    }
}
