//! An abstraction for worker threads.
//!
//! The abstraction provided some consistency on how worker threads
//! are implemented and used, as well as logging. A worker thread does
//! what it does. The abstraction is not meant to constrain that.

use std::thread::{JoinHandle, spawn};

/// Start a new thread. Caller must catch the thread handle and
/// join it to wait for thread to end.
pub fn start_thread<W: Worker>(mut o: W) -> JoinHandle<Result<(), W::Error>> {
    let name = o.name();
    spawn(move || {
        eprintln!("start worker {name}");
        let result = o.work();
        eprintln!("end worker {name}: result={result:?}");
        result
    })
}

/// A worker thread.
pub trait Worker: Send + 'static {
    /// Name of thread, or kind of thread. Used for logging only.
    const NAME: &str;

    /// Type of error from this worker.
    type Error: std::fmt::Debug + Send;

    /// Do the work the thread is supposed to do.
    fn work(&mut self) -> Result<(), Self::Error>;

    /// Return name of thread as an owned string.
    fn name(&self) -> String {
        Self::NAME.to_string()
    }
}
