{
  description = "Radicle CI Broker";

  inputs = {
    nixpkgs.url = "github:NixOS/nixpkgs/release-24.11";

    crane = {
      url = "github:ipetkov/crane";
    };

    rust-overlay = {
      url = "github:oxalica/rust-overlay";
      inputs = {
        nixpkgs.follows = "nixpkgs";
      };
    };

    radicle = {
      url = "git+https://seed.radicle.xyz/rad:z3gqcJUoA1n9HaHKufZs5FCSGazv5.git?tag=v1.1.0";
    };

    flake-utils.url = "github:numtide/flake-utils";

    advisory-db = {
      url = "github:rustsec/advisory-db";
      flake = false;
    };
  };

  nixConfig = {
    keepOutputs = true;
  };

  outputs = {
    self,
    nixpkgs,
    crane,
    flake-utils,
    advisory-db,
    radicle,
    rust-overlay,
    ...
  }:
    flake-utils.lib.eachDefaultSystem (system: let
      pname = "radicle-ci-broker";
      lib = nixpkgs.lib;
      pkgs = import nixpkgs {
        inherit system;
        overlays = [(import rust-overlay)];
      };

      rustToolChain = pkgs.rust-bin.fromRustupToolchainFile ./rust-toolchain;
      craneLib = (crane.mkLib pkgs).overrideToolchain rustToolChain;

      srcFilters = path: type:
        builtins.any (suffix: lib.hasSuffix suffix path) [
          ".subplot" # build and documentation
          ".md"
          ".yaml"
          ".css"
        ]
        ||
        # Default filter from crane (allow .rs files)
        (craneLib.filterCargoSources path type);

      src = lib.cleanSourceWith {
        src = ./.;
        filter = srcFilters;
      };

      basicArgs = {
        inherit src;
        inherit pname;
        strictDeps = true;
      };

      # Build *just* the cargo dependencies, so we can reuse
      # all of that work (e.g. via cachix) when running in CI
      cargoArtifacts = craneLib.buildDepsOnly basicArgs;

      # Common arguments can be set here to avoid repeating them later
      commonArgs =
        basicArgs
        // {
          inherit cargoArtifacts;

          nativeBuildInputs = with pkgs; [
            git
            # Add additional build inputs here
          ];
          buildInputs = lib.optionals pkgs.stdenv.buildPlatform.isDarwin (with pkgs; [
            darwin.apple_sdk.frameworks.Security
          ]);
        };

      radicle-ci-broker = craneLib.buildPackage (commonArgs
        // {
          inherit (craneLib.crateNameFromCargoToml {cargoToml = ./Cargo.toml;}) pname version;
          doCheck = false;
        });

      bins = let
        bin = {
          name,
        }:
          craneLib.buildPackage (commonArgs
            // {
              inherit (craneLib.crateNameFromCargoToml {cargoToml = ./Cargo.toml;}) pname version;
              cargoExtraArgs = "--bin ${name}";
              doCheck = false;
            });
        bins = builtins.listToAttrs (map
          ({name, ...} @ package: lib.nameValuePair name (bin package))
          [
            {
              name = "cibtool";
            }
            {
              name = "cib";
            }
            {
              name = "synthetic-events";
            }
          ]);
      in
        bins
        // rec {
          default = radicle-ci-broker;
          ci-broker = pkgs.buildEnv {
            name = "ci-broker";
            paths = with bins; [
              cibtool
              cib
              synthetic-events
            ];
          };
        };
    in {
      # Formatter
      formatter = pkgs.alejandra;

      # Set of checks that are run: `nix flake check`
      checks = {
        # Run clippy (and deny all warnings) on the crate source,
        # again, reusing the dependency artifacts from above.
        #
        # Note that this is done as a separate derivation so that
        # we can block the CI if there are issues here, but not
        # prevent downstream consumers from building our crate by itself.
        clippy = craneLib.cargoClippy (commonArgs
          // {
            cargoClippyExtraArgs = "--all-targets -- --deny warnings";
          });

        doc = craneLib.cargoDoc commonArgs;
        fmt = craneLib.cargoFmt basicArgs;
        deny = craneLib.cargoDeny commonArgs;

        audit = craneLib.cargoAudit {
          inherit src advisory-db;
        };

        # Run tests with cargo-nextest
        nextest = craneLib.cargoNextest (commonArgs
          // {
            # We skip the test since it uses the underlying .git directory,
            # which is not available in the Nix sandbox.
            # In any case, this test is slow and we expect it to be tested
            # before merges (and it can be tested in the devShell)
            cargoNextestExtraArgs = "-- --skip acceptance_criteria_for_upgrades";
            partitions = 1;
            partitionType = "count";
            nativeBuildInputs = [
              pkgs.bash
              pkgs.git
              pkgs.jq
              pkgs.sqlite
              radicle.packages.${system}.radicle-cli
              radicle.packages.${system}.radicle-remote-helper
            ];

            # Ensure dev is used since we rely on env variables being
            # set in tests.
            env.CARGO_PROFILE = "dev";
          });
      };

      packages.default = radicle-ci-broker;

      apps.cibtool = flake-utils.lib.mkApp {
        drv = self.bins.${system}.cibtool;
      };

      apps.cib = flake-utils.lib.mkApp {
        drv = self.bins.${system}.cib;
      };

      apps.synthetic-events = flake-utils.lib.mkApp {
        drv = self.bins.${system}.synthetic-events;
      };

      devShells.default = craneLib.devShell {
        # Extra inputs can be added here; cargo and rustc are provided by default.
        packages = with pkgs; [
          bash
          coreutils
          cargo-watch
          cargo-nextest
          git
          jq
          ripgrep
          rust-analyzer
          sqlite
        ];

        env.RUST_SRC_PATH = "${rustToolChain}/lib/rustlib/src/rust/library";
      };
    });
}

