use radicle::Profile;

use crate::{
    ci_event::CiEvent,
    logger,
    node_event_source::{NodeEventError, NodeEventSource},
};

pub struct CiEventSource {
    source: NodeEventSource,
}

impl CiEventSource {
    pub fn new(profile: &Profile) -> Result<Self, CiEventSourceError> {
        let source = NodeEventSource::new(profile).map_err(CiEventSourceError::Subscribe)?;
        Ok(Self { source })
    }

    pub fn event(&mut self) -> Result<Vec<CiEvent>, CiEventSourceError> {
        let result = self.source.node_event();
        match &result {
            Err(NodeEventError::BrokenConnection) => {
                logger::event_disconnected();
                Err(CiEventSourceError::BrokenConnection(result.unwrap_err()))
            }
            Err(err) => {
                logger::error("error reading event from node", &err);
                Err(CiEventSourceError::NodeEventError(result.unwrap_err()))
            }
            Ok(None) => {
                logger::event_end();
                Ok(vec![])
            }
            Ok(Some(event)) => CiEvent::from_node_event(event).map_err(CiEventSourceError::CiEvent),
        }
    }
}

#[derive(Debug, thiserror::Error)]
pub enum CiEventSourceError {
    #[error("failed to subscribe to node events")]
    Subscribe(#[source] crate::node_event_source::NodeEventError),

    #[error("connection to node control socket broke")]
    BrokenConnection(#[source] crate::node_event_source::NodeEventError),

    #[error("failed to read event from node")]
    NodeEventError(#[source] crate::node_event_source::NodeEventError),

    #[error("failed to create CI events from node event")]
    CiEvent(#[source] crate::ci_event::CiEventError),
}
