//! A logger abstraction for the CI broker.

#[cfg(test)]
use std::sync::Once;
use std::{path::Path, sync::Mutex, time::Duration};

use radicle::{git::raw::Oid, identity::RepoId};
use slog::{debug, error, info, o, warn, Drain};
use slog_scope::GlobalLoggerGuard;

use crate::{ci_event::CiEvent, config::Config, db::QueueId, msg::Request, run::Run};

pub fn open() -> GlobalLoggerGuard {
    let logger = slog_json::Json::new(std::io::stderr())
        .add_default_keys()
        .set_flush(true)
        .set_newlines(true)
        .build();
    let log = slog::Logger::root(Mutex::new(logger).fuse(), o!());
    slog_scope::set_global_logger(log)
}

// Set up structured logging for tests.
//
// We have to keep the GlobalLoggerGuard we get when we initialize
// `slog-scope` alive as long as we may need to do any logging. We can
// only create that once per process. For tests, we do that here.
//
// We can't do the same thing for non-test processes, as that would
// interfere with use of `radicle-ci-broker` as a library. Libraries
// should not interfere with global state, unless they're specifically
// intended to do that.
//
// This is for tests only: otherwise

// We use this to make sure we initialize the logger only once.
#[cfg(test)]
static INIT: Once = Once::new();

// This is the actual logger for tests.
#[cfg(test)]
static mut LOGGER: Option<GlobalLoggerGuard> = None;

#[cfg(test)]
#[ctor::ctor]
fn open_for_tests() {
    INIT.call_once(|| unsafe {
        LOGGER = Some(open());
    });
}

pub fn start_cib() {
    info!(slog_scope::logger(), "CI broker starts"; "version" => env!("GIT_HEAD"));
}

pub fn end_cib_successfully() {
    info!(slog_scope::logger(), "CI broker ends successfully");
}

pub fn end_cib_in_error() {
    info!(
        slog_scope::logger(),
        "CI broker ends in unrecoverable error"
    );
}

pub fn loaded_config(config: &Config) {
    debug!(slog_scope::logger(), "loaded configuration {config:#?}");
}

pub fn queueproc_start() {
    info!(
        slog_scope::logger(),
        "start thread to process events until a shutdown event"
    );
}

pub fn queueproc_end() {
    info!(slog_scope::logger(), "thread to process events ends");
}

pub fn queueproc_channel_disconnect() {
    info!(
        slog_scope::logger(),
        "event notification channel disconnected"
    );
}

pub fn queueproc_picked_event(id: &QueueId) {
    info!(slog_scope::logger(), "picked event from queue: {id}");
}

pub fn queueproc_remove_event(id: &QueueId) {
    info!(slog_scope::logger(), "remove event from queue: {id}");
}

pub fn queueproc_action_run(rid: &RepoId, oid: &Oid) {
    info!(slog_scope::logger(), "Action: run: {rid} {oid}");
}

pub fn queueproc_action_shutdown() {
    info!(slog_scope::logger(), "Action: shutdown");
}

pub fn queueadd_start() {
    info!(
        slog_scope::logger(),
        "start thread to add events from node to event queue"
    );
}

pub fn queueadd_control_socket_close() {
    info!(
        slog_scope::logger(),
        "no more events from node control sockets"
    );
}

pub fn queueadd_push_event(e: &CiEvent) {
    debug!(
        slog_scope::logger(),
        "insert broker event into queue: {e:?}"
    );
}

pub fn queueadd_end() {
    info!(slog_scope::logger(), "start thread to process events ends");
}

pub fn pages_directory_unset() {
    warn!(
        slog_scope::logger(),
        "not writing HTML report pages as output directory has not been set"
    );
}

pub fn pages_interval(interval: Duration) {
    info!(
        slog_scope::logger(),
        "wait about {} seconds to update HTML report pages again",
        interval.as_secs()
    );
}

pub fn pages_disconnected() {
    info!(
        slog_scope::logger(),
        "page updater: run notification channel disconnected"
    );
}

pub fn pages_start() {
    info!(slog_scope::logger(), "start page updater thread");
}

pub fn pages_end() {
    info!(slog_scope::logger(), "end page updater thread");
}

pub fn event_disconnected() {
    info!(
        slog_scope::logger(),
        "connection to node control socket broke"
    );
}

pub fn event_end() {
    info!(
        slog_scope::logger(),
        "no more node events from control socket: iterator ended"
    );
}

pub fn broker_db(filename: &Path) {
    info!(
        slog_scope::logger(),
        "broker database: {}",
        filename.display()
    );
}

pub fn broker_start_run(trigger: &Request) {
    info!(slog_scope::logger(), "start CI run");
    debug!(slog_scope::logger(), "trigger event: {trigger:#?}");
}

pub fn broker_end_run(run: &Run) {
    info!(slog_scope::logger(), "Finish CI run");
    debug!(slog_scope::logger(), "finished CI run: {run:#?}");
}

pub fn adapter_no_first_response() {
    error!(slog_scope::logger(), "no first response message");
}

pub fn adapter_no_second_response() {
    error!(slog_scope::logger(), "no second response message");
}

pub fn adapter_too_many_responses() {
    error!(slog_scope::logger(), "too many response messages");
}

pub fn adapter_result(exit: Option<i32>, stderr: &str) {
    if let Some(exit) = exit {
        debug!(slog_scope::logger(), "adapter exit code"; "exit_code" => exit);
    } else {
        debug!(slog_scope::logger(), "adapter was terminated by signal");
    }
    for line in stderr.lines() {
        debug!(slog_scope::logger(), "adapter stderr"; "stderr" => line);
    }
}

pub fn adapter_did_not_exit_voluntarily() {
    warn!(slog_scope::logger(), "adapter did not exit voluntarily");
}

pub fn debug(msg: &str) {
    debug!(slog_scope::logger(), "{msg}");
}

pub fn debug2(msg: String) {
    debug!(slog_scope::logger(), "{msg}");
}

pub fn error(msg: &str, e: &impl std::error::Error) {
    error!(slog_scope::logger(), "{msg}: {e}");
    let mut e = e.source();
    while let Some(source) = e {
        error!(slog_scope::logger(), "caused by: {}", source);
        e = source.source();
    }
}
