//! Notification channel between threads.

use std::sync::mpsc::{channel, Receiver, Sender};

/// Channel endpoint for sending notifications.
pub type NotificationSender = Sender<()>;

/// Channel endpoint for receiving notifications.
pub type NotificationReceiver = Receiver<()>;

/// Notification channel.
///
/// The notification channel allows one thread to notify another that
/// the other thread has some work to do. The notification carries no
/// other information: the receiver is supposed to know where it can
/// get whatever data it needs to what it needs to do.
///
/// The point of this is to make sure threads in the CI broker
/// exchange data only via the database, where it is persistent.
pub struct NotificationChannel {
    tx: Option<NotificationSender>,
    rx: Option<NotificationReceiver>,
}

impl Default for NotificationChannel {
    fn default() -> Self {
        let (tx, rx) = channel();
        Self {
            tx: Some(tx),
            rx: Some(rx),
        }
    }
}

impl NotificationChannel {
    /// Return the transmit endpoint of the notification channel. This
    /// can only be called once.
    pub fn tx(&mut self) -> NotificationSender {
        self.tx.take().unwrap()
    }

    /// Return the receive endpoint of the notification channel. This
    /// can only be called once.
    pub fn rx(&mut self) -> NotificationReceiver {
        self.rx.take().unwrap()
    }
}
