// Implementations of Subplot scenario steps for the CI broker.

use std::{
    fs::{set_permissions, Permissions},
    io::Write,
    os::unix::fs::PermissionsExt,
    path::{Path, PathBuf},
};

use subplotlib::steplibrary::datadir::Datadir;
use subplotlib::steplibrary::files::Files;
use subplotlib::steplibrary::runcmd::Runcmd;

#[derive(Debug, Default)]
struct SubplotContext {}

impl ContextElement for SubplotContext {}

#[step]
#[context(SubplotContext)]
#[context(Runcmd)]
fn install_ci_broker(context: &ScenarioContext) {
    let target_path = bindir();
    assert!(target_path.join("cib").exists());
    assert!(target_path.join("cibtool").exists());
    assert!(target_path.join("synthetic-events").exists());
    context.with_mut(
        |context: &mut Runcmd| {
            context.prepend_to_path(target_path);
            Ok(())
        },
        false,
    )?;
}

fn bindir() -> PathBuf {
    let path = if let Ok(target) = std::env::var("CARGO_TARGET_DIR") {
        Path::new(&target).join("debug")
    } else {
        PathBuf::from("target/debug")
    };
    path.canonicalize().unwrap()
}

#[step]
#[context(SubplotContext)]
#[context(Datadir)]
#[context(Files)]
fn install_adapter(context: &Datadir, filename: &Path, embedded: SubplotDataFile) {
    eprintln!(
        "install adapter {} from {}:\n{}",
        filename.display(),
        embedded.name().display(),
        String::from_utf8_lossy(embedded.data()),
    );
    eprintln!("write {}", filename.display());
    context.open_write(filename)?.write_all(embedded.data())?;
    let realpath = context.canonicalise_filename(filename)?;
    eprintln!(
        "chmod {} (exists? {})",
        realpath.display(),
        realpath.exists()
    );
    let executable = Permissions::from_mode(0o755);
    set_permissions(realpath, executable)?;
}

#[step]
#[context(SubplotContext)]
#[context(Runcmd)]
fn stdout_has_one_line(runcmd: &Runcmd) {
    let linecount = runcmd.stdout_as_string().lines().count();
    if linecount != 1 {
        throw!(format!("stdout had {linecount} lines, expected 1"));
    }
}

#[step]
#[context(SubplotContext)]
#[context(Runcmd)]
fn stdout_is_empty(runcmd: &Runcmd) {
    let stdout = runcmd.stdout_as_string();
    if !stdout.is_empty() {
        throw!(format!(
            "expected stdout to be empty, is actually {stdout:?}"
        ));
    }
}
