use std::{
    fs::{File, OpenOptions},
    io::Write,
    path::{Path, PathBuf},
};

#[derive(Debug)]
pub struct AdminLog {
    filename: PathBuf,
    file: File,
}

impl AdminLog {
    pub fn open(filename: &Path) -> Result<Self, LogError> {
        let file = OpenOptions::new()
            .append(true)
            .create(true)
            .open(filename)
            .map_err(|e| LogError::OpenLogFile(filename.into(), e))?;
        Ok(Self {
            filename: filename.into(),
            file,
        })
    }

    pub fn h1(&mut self, text: &str) -> Result<(), LogError> {
        self.write("\n# ")?;
        self.write(text)?;
        self.write("\n\n")?;
        Ok(())
    }

    pub fn h2(&mut self, text: &str) -> Result<(), LogError> {
        self.write("\n## ")?;
        self.write(text)?;
        self.write("\n\n")?;
        Ok(())
    }

    pub fn h3(&mut self, text: &str) -> Result<(), LogError> {
        self.write("\n### ")?;
        self.write(text)?;
        self.write("\n\n")?;
        Ok(())
    }

    pub fn bullet_point(&mut self, text: String) -> Result<(), LogError> {
        self.write(&text)?;
        self.write("\n")?;
        Ok(())
    }

    pub fn writeln(&mut self, text: &str) -> Result<(), LogError> {
        self.write(text)?;
        self.write("\n")?;
        Ok(())
    }

    pub fn fenced(&mut self, msg: &str, data: &[u8]) -> Result<(), LogError> {
        const FENCED_BLOCK: &str = "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n";
        self.h3(msg)?;
        self.write(FENCED_BLOCK)?;
        if !data.is_empty() {
            let text = String::from_utf8_lossy(data);
            self.write(&text)?;
            if !text.ends_with('\n') {
                self.write("\n")?;
            }
        }
        self.write(FENCED_BLOCK)?;
        self.write("\n")?;

        Ok(())
    }

    fn write(&mut self, msg: &str) -> Result<(), LogError> {
        self.file
            .write_all(msg.as_bytes())
            .map_err(|e| LogError::WriteLogFile(self.filename.clone(), e))
    }
}

#[derive(Debug, thiserror::Error)]
pub enum LogError {
    #[error("failed to open log file {0}")]
    OpenLogFile(PathBuf, #[source] std::io::Error),

    #[error("failed to write to log file {0}")]
    WriteLogFile(PathBuf, #[source] std::io::Error),
}
