use std::path::PathBuf;

use radicle_ci_broker::msg::{MessageError, Request, Response, RunId, RunResult};

use crate::{config::ConfigError, logfile::LogError, runspec::RunSpecError};

/// Read a request from stdin.
pub fn read_request() -> Result<Request, NativeMessageError> {
    let req = Request::from_reader(std::io::stdin()).map_err(NativeMessageError::ReadRequest)?;
    Ok(req)
}

// Write response to stdout.
fn write_response(resp: &Response) -> Result<(), NativeMessageError> {
    resp.to_writer(std::io::stdout())
        .map_err(|e| NativeMessageError::WriteResponse(resp.clone(), Box::new(e)))?;
    Ok(())
}

/// Write a "triggered" response to stdout.
pub fn write_triggered(run_id: &RunId, info_url: Option<&str>) -> Result<(), NativeMessageError> {
    let response = if let Some(url) = info_url {
        Response::triggered_with_url(run_id.clone(), url)
    } else {
        Response::triggered(run_id.clone())
    };
    write_response(&response)?;
    Ok(())
}

/// Write a message indicating failure to stdout.
pub fn write_failed() -> Result<(), NativeMessageError> {
    write_response(&Response::Finished {
        result: RunResult::Failure,
    })?;
    Ok(())
}

/// Write a message indicating success to stdout.
pub fn write_succeeded() -> Result<(), NativeMessageError> {
    write_response(&Response::Finished {
        result: RunResult::Success,
    })?;
    Ok(())
}

#[derive(Debug, thiserror::Error)]
pub enum NativeMessageError {
    #[error("failed to read request from stdin: {0:?}")]
    ReadRequest(#[source] MessageError),

    #[error("failed to write response to stdout: {0:?}")]
    WriteResponse(Response, #[source] Box<MessageError>),

    #[error("failed to create per-run parent directory {0}")]
    CreateState(PathBuf, #[source] std::io::Error),

    #[error("failed to create per-run directory {0}")]
    CreateRunDir(PathBuf, #[source] std::io::Error),

    #[error("failed to load Radicle profile")]
    LoadProfile(#[source] radicle::profile::Error),

    #[error(transparent)]
    Config(#[from] ConfigError),

    #[error(transparent)]
    Log(#[from] LogError),

    #[error(transparent)]
    RunSpec(#[from] RunSpecError),
}
