use std::{
    ffi::{OsStr, OsString},
    process::{Command, Output, exit},
    thread::sleep,
    time::Duration,
};

use clap::Parser;
use clingwrap::runner::{CommandError, CommandRunner};
use indicatif::{MultiProgress, ProgressBar, ProgressStyle};

const EXIT_KILLED: i32 = 1;
const EXIT_IMPOSSIBLE: i32 = 2;
const EXIT_UNAVAILBLE: i32 = 3;

fn main() -> Result<(), std::io::Error> {
    let args = Args::parse();

    let mut progress = if args.until {
        Progress::until_succeeds(&args)
    } else {
        Progress::until_fails(&args)
    };

    if args.quiet {
        progress.be_quiet();
    }

    match args.mode() {
        Mode::UntilHappy => match until_happy(&progress, args.wait, &args.argv0, &args.args) {
            Outcome::Exit(code) => exit(code),
            Outcome::Happy => (),
        },
        Mode::UntilUnhappy => match until_unhappy(&progress, args.wait, &args.argv0, &args.args) {
            Outcome::Exit(code) => exit(code),
            Outcome::Happy => (),
        },
    }

    progress.finish();
    println!(
        "repeatcmd ends in glorious success after {} tries to run the command",
        progress.attempts(),
    );
    Ok(())
}

fn until_happy(progress: &Progress, wait: u64, argv0: &OsStr, args: &[OsString]) -> Outcome {
    let wait = Duration::from_secs(wait);
    loop {
        let res = once(progress, argv0, args);
        match res {
            CommandResult::Failed => sleep(wait),
            CommandResult::Success => return Outcome::Happy,
            CommandResult::Impossible => return Outcome::Exit(EXIT_IMPOSSIBLE),
            CommandResult::Killed => return Outcome::Exit(EXIT_KILLED),
            CommandResult::Unavailable => return Outcome::Exit(EXIT_UNAVAILBLE),
        }
    }
}

fn until_unhappy(progress: &Progress, wait: u64, argv0: &OsStr, args: &[OsString]) -> Outcome {
    let wait = Duration::from_secs(wait);
    loop {
        let res = once(progress, argv0, args);
        match res {
            CommandResult::Failed => return Outcome::Happy,
            CommandResult::Success => sleep(wait),
            CommandResult::Impossible => return Outcome::Exit(EXIT_IMPOSSIBLE),
            CommandResult::Killed => return Outcome::Exit(EXIT_KILLED),
            CommandResult::Unavailable => return Outcome::Exit(EXIT_UNAVAILBLE),
        }
    }
}

fn once(progress: &Progress, argv0: &OsStr, args: &[OsString]) -> CommandResult {
    progress.start();

    let mut cmd = Command::new(argv0);
    cmd.args(args);

    let mut runner = CommandRunner::new(cmd);
    runner.capture_stdout();
    runner.capture_stderr();

    match runner.execute() {
        Ok(output) => {
            progress.output(&output);
            progress.end("command was successful".to_string());
            CommandResult::Success
        }

        Err(CommandError::NoSuchCommand(_)) => {
            progress.end("no such command".to_string());
            CommandResult::Unavailable
        }

        Err(CommandError::NoPermission(_)) => {
            progress.end("no permission to execute".to_string());
            CommandResult::Unavailable
        }

        Err(CommandError::CommandFailed {
            exit_code, output, ..
        }) => {
            progress.output(&output);
            progress.end(format!("exit with code {exit_code}"));
            CommandResult::Failed
        }

        Err(CommandError::KilledBySignal { signal, .. }) => {
            progress.end(format!("terminated by signal {signal}"));
            CommandResult::Killed
        }

        Err(err) => {
            progress.end(format!("{err}"));
            CommandResult::Impossible
        }
    }
}

/// Run command repeatedly until it succeeds.
///
/// If command exits with a non-zero exit code, it is run again. If it
/// exits with zero, or terminates due to a signal, it is not run
/// again. The command is NOT given to the shell to execute. It is
/// invoked directly.
#[derive(Debug, Parser)]
struct Args {
    /// Run command repeatedly until it succeeds; this is the default mode of operation.
    #[clap(long)]
    until: bool,

    /// Run command repeatedly until it fails; this is the default mode of operation.
    #[clap(long)]
    until_fails: bool,

    /// Wait this many seconds before trying again after command fails.
    #[clap(long, short = 'n', default_value = "1")]
    wait: u64,

    /// Don't show output from command.
    #[clap(long)]
    quiet: bool,

    /// Command to run.
    argv0: OsString,

    /// Arguments to command.
    args: Vec<OsString>,
}

impl Args {
    fn mode(&self) -> Mode {
        if self.until_fails {
            Mode::UntilUnhappy
        } else {
            Mode::UntilHappy
        }
    }
}

enum Mode {
    UntilHappy,
    UntilUnhappy,
}

enum CommandResult {
    Success,
    Unavailable,
    Failed,
    Killed,
    Impossible,
}

enum Outcome {
    Happy,
    Exit(i32),
}

struct Progress {
    multi: MultiProgress,
    previous_result: ProgressBar,
    attempt: ProgressBar,
    quiet: bool,
}

impl Progress {
    fn until_succeeds(args: &Args) -> Self {
        let top_bar =
            ProgressBar::no_length().with_style(ProgressStyle::with_template("{msg}").unwrap());
        top_bar.set_message(format!(
            "repeat until succeeds: {:?} {:?}",
            args.argv0, args.args
        ));
        Self::new(top_bar)
    }

    fn until_fails(args: &Args) -> Self {
        let top_bar =
            ProgressBar::no_length().with_style(ProgressStyle::with_template("{msg}").unwrap());
        top_bar.set_message(format!(
            "repeat until fails: {:?} {:?}",
            args.argv0, args.args
        ));
        Self::new(top_bar)
    }

    fn new(top_bar: ProgressBar) -> Self {
        let multi = MultiProgress::new();
        multi.add(top_bar);

        let previous_result = ProgressBar::no_length()
            .with_style(ProgressStyle::with_template("previous result: {msg}").unwrap());
        multi.add(previous_result.clone());

        let attempt = ProgressBar::no_length()
            .with_style(ProgressStyle::with_template("{elapsed}; attempt #{pos}").unwrap());
        multi.add(attempt.clone());

        Self {
            multi,
            previous_result,
            attempt,
            quiet: false,
        }
    }

    fn be_quiet(&mut self) {
        self.quiet = true;
    }

    fn start(&self) {
        self.attempt.inc(1);
    }

    fn end(&self, msg: String) {
        self.previous_result.set_message(msg);
    }

    fn attempts(&self) -> u64 {
        self.attempt.position()
    }

    fn output(&self, output: &Output) {
        fn helper(multi: &MultiProgress, stream: &'static str, bytes: &[u8]) {
            if !bytes.is_empty() {
                multi.println(stream).ok();
                for line in String::from_utf8_lossy(bytes).lines() {
                    let line = format!("  {line}");
                    multi.println(line).ok();
                }
            }
        }

        if !self.quiet {
            helper(&self.multi, "stdout:", &output.stdout);
            helper(&self.multi, "stderr:", &output.stderr);
        }
    }

    fn finish(&self) {
        self.multi.clear().ok();
    }
}
