use std::{fs::create_dir_all, path::PathBuf};

use clap::Parser;
use log::{debug, info};

use crate::{
    store::{EncryptedStore, EncryptedStoreError, Store, StoreError},
    Config, LeafCommand,
};

#[derive(Debug, Parser)]
pub struct InitCommand {
    #[clap(long)]
    name: String,

    #[clap(long)]
    key: PathBuf,
}

impl LeafCommand for InitCommand {
    type Error = InitError;

    fn run(&self, config: &Config) -> Result<(), InitError> {
        info!("init password store");
        debug!("{config:#?}");
        let store_dir = config.store();
        if !store_dir.exists() {
            println!("creating {}", store_dir.display());
            create_dir_all(store_dir).map_err(|err| InitError::CreateDir(store_dir.into(), err))?;
        }

        let encrypted = EncryptedStore::new(config.sop(), config.store());
        encrypted.install_key(&self.key)?;
        let cert = encrypted.extract_cert()?;

        let mut store = Store::default();
        store.push_cert(&self.name, &cert);

        encrypted.write_store(&store)?;

        Ok(())
    }
}

#[derive(Debug, thiserror::Error)]
pub enum InitError {
    #[error("failed to create store directory {0}")]
    CreateDir(PathBuf, #[source] std::io::Error),

    #[error("failed to write store to file {0}")]
    Write(PathBuf, #[source] StoreError),

    #[error(transparent)]
    Encrypted(#[from] EncryptedStoreError),
}
