use std::{error::Error, path::PathBuf, process::exit};

use clap::Parser;
use directories_next::ProjectDirs;
use log::info;

use sopass::{cmd, config::Config, LeafCommand};

const QUAL: &str = "";
const ORG: &str = "";
const APP: &str = "sopass";

fn main() {
    if let Err(err) = fallible_main() {
        eprintln!("ERROR: {err}");
        let mut err = err.source();
        while let Some(underlying) = err {
            eprintln!("caused by: {underlying}");
            err = underlying.source();
        }
        exit(1);
    }
}

fn fallible_main() -> Result<(), SopassError> {
    let args = Args::parse();
    env_logger::init_from_env("SOPASS_LOG");
    let config = args.config();

    info!("sopass starts");
    info!("store is {}", config.store().display());

    info!("main command start");
    match &args.cmd {
        Command::Cert(x) => x.run(&config)?,
        Command::Config(x) => x.run(&config)?,
        Command::Init(x) => x.run(&config)?,
        Command::Key(x) => x.run(&config)?,
        Command::Value(x) => x.run(&config)?,
        Command::Version(x) => x.run(&config)?,
    }

    info!("main command OK");
    Ok(())
}

#[derive(Debug, Parser)]
#[command(version)]
struct Args {
    #[clap(long)]
    store: Option<PathBuf>,

    #[clap(long, default_value = "rsop")]
    sop: PathBuf,

    #[clap(subcommand)]
    cmd: Command,
}

impl Args {
    fn store(&self) -> PathBuf {
        self.store.clone().unwrap_or_else(default_store)
    }

    fn config(&self) -> Config {
        Config::new(&self.store(), &self.sop)
    }
}

fn default_store() -> PathBuf {
    if let Some(dirs) = ProjectDirs::from(QUAL, ORG, APP) {
        dirs.data_dir().into()
    } else {
        panic!("failed to construct default store path");
    }
}

#[derive(Debug, Parser)]
enum Command {
    Cert(CertCommand),
    Config(cmd::ConfigCommand),
    Init(cmd::InitCommand),
    Key(KeyCommand),
    Value(ValueCommand),
    Version(cmd::VersionCommand),
}

#[derive(Debug, Parser)]
struct CertCommand {
    #[clap(subcommand)]
    cmd: CertSubcommand,
}

impl CertCommand {
    fn run(&self, config: &Config) -> Result<(), SopassError> {
        match &self.cmd {
            CertSubcommand::Add(x) => x.run(config)?,
            CertSubcommand::List(x) => x.run(config)?,
        }
        Ok(())
    }
}

#[derive(Debug, Parser)]
enum CertSubcommand {
    Add(cmd::AddCert),
    List(cmd::ListCerts),
}

#[derive(Debug, Parser)]
struct KeyCommand {
    #[clap(subcommand)]
    cmd: KeySubcommand,
}

impl KeyCommand {
    fn run(&self, config: &Config) -> Result<(), SopassError> {
        match &self.cmd {
            KeySubcommand::Extract(x) => x.run(config)?,
        }
        Ok(())
    }
}

#[derive(Debug, Parser)]
enum KeySubcommand {
    Extract(cmd::ExtractCert),
}

#[derive(Debug, Parser)]
struct ValueCommand {
    #[clap(subcommand)]
    cmd: ValueSubcommand,
}

impl ValueCommand {
    fn run(&self, config: &Config) -> Result<(), SopassError> {
        info!("value command start");
        match &self.cmd {
            ValueSubcommand::Add(x) => x.run(config)?,
            ValueSubcommand::List(x) => x.run(config)?,
            ValueSubcommand::Remove(x) => x.run(config)?,
            ValueSubcommand::Rename(x) => x.run(config)?,
            ValueSubcommand::Show(x) => x.run(config)?,
        }
        info!("value command OK");
        Ok(())
    }
}

#[derive(Debug, Parser)]
enum ValueSubcommand {
    Add(cmd::AddValue),
    List(cmd::ListValues),
    Remove(cmd::RemoveValue),
    Rename(cmd::RenameValue),
    Show(cmd::ShowValue),
}

#[derive(Debug, thiserror::Error)]
enum SopassError {
    #[error(transparent)]
    Cert(#[from] cmd::CertError),

    #[error(transparent)]
    Config(#[from] cmd::ConfigError),

    #[error(transparent)]
    Init(#[from] cmd::InitError),

    #[error(transparent)]
    Key(#[from] cmd::KeyError),

    #[error(transparent)]
    Value(#[from] cmd::ValueError),

    #[error(transparent)]
    Version(#[from] cmd::VersionError),
}
