use std::{
    fs::File,
    io::Write,
    path::{Path, PathBuf},
    process::{Command, Stdio},
    thread::spawn,
};

use log::info;
use tempfile::tempdir;

pub struct Key {
    data: Vec<u8>,
}

impl Key {
    pub fn new(data: Vec<u8>) -> Self {
        Self { data }
    }

    pub fn as_bytes(&self) -> &[u8] {
        &self.data
    }
}

pub struct Certificate {
    data: Vec<u8>,
}

impl Certificate {
    pub fn new(data: Vec<u8>) -> Self {
        Self { data }
    }

    pub fn as_bytes(&self) -> &[u8] {
        &self.data
    }
}

pub struct Sop {
    sop: PathBuf,
    key: PathBuf,
}

impl Sop {
    pub fn new(sop: &Path, key: &Path) -> Self {
        Self {
            sop: sop.into(),
            key: key.into(),
        }
    }

    pub fn extract_cert(&self) -> Result<Certificate, SopError> {
        let mut child = Command::new(&self.sop)
            .arg("extract-cert")
            .stdin(Stdio::piped())
            .stdout(Stdio::piped())
            .spawn()
            .map_err(|err| SopError::Invoke(self.sop.clone(), err))?;

        let key = self.key()?;
        let mut stdin = child.stdin.take().ok_or(SopError::TakeStdin)?;
        let writer = spawn(move || stdin.write_all(key.as_bytes()));
        writer
            .join()
            .map_err(|_| SopError::JoinThread)?
            .map_err(SopError::WriteStdin)?;

        let output = child.wait_with_output().map_err(SopError::WaitChild)?;

        if !output.status.success() {
            let stderr = String::from_utf8_lossy(&output.stderr).to_string();
            return Err(SopError::Failed(
                self.sop.clone(),
                output.status.code().unwrap_or(999),
                stderr,
            ));
        }

        Ok(Certificate::new(output.stdout))
    }

    pub fn encrypt(
        &self,
        data: Vec<u8>,
        certs: &[Certificate],
        output: &Path,
    ) -> Result<Vec<u8>, SopError> {
        info!("encrypt data to {}", output.display());

        let tmp = tempdir().map_err(SopError::TempDir)?;

        let mut filenames = vec![];
        for (i, cert) in certs.iter().enumerate() {
            let filename = format!("cert-{i}");
            let filename = tmp.path().join(&filename);
            std::fs::write(&filename, cert.as_bytes()).map_err(SopError::TempWrite)?;
            filenames.push(PathBuf::from(&filename));
        }

        let stdout =
            File::create(output).map_err(|err| SopError::CreateFile(output.into(), err))?;

        let mut child = Command::new(&self.sop)
            .arg("encrypt")
            .args(&filenames)
            .stdin(Stdio::piped())
            .stdout(stdout)
            .spawn()
            .map_err(|err| SopError::Invoke(self.sop.clone(), err))?;

        let mut stdin = child.stdin.take().ok_or(SopError::TakeStdin)?;
        let writer = spawn(move || stdin.write_all(&data));
        writer
            .join()
            .map_err(|_| SopError::JoinThread)?
            .map_err(SopError::WriteStdin)?;

        let output = child.wait_with_output().map_err(SopError::WaitChild)?;

        if !output.status.success() {
            let stderr = String::from_utf8_lossy(&output.stderr).to_string();
            return Err(SopError::Failed(
                self.sop.clone(),
                output.status.code().unwrap_or(999),
                stderr,
            ));
        }

        Ok(output.stdout)
    }

    pub fn decrypt(&self, data: Vec<u8>) -> Result<Vec<u8>, SopError> {
        let mut child = Command::new(&self.sop)
            .arg("decrypt")
            .arg(&self.key)
            .stdin(Stdio::piped())
            .stdout(Stdio::piped())
            .spawn()
            .map_err(|err| SopError::Invoke(self.sop.clone(), err))?;

        let mut stdin = child.stdin.take().ok_or(SopError::TakeStdin)?;
        let writer = spawn(move || stdin.write_all(&data));
        writer
            .join()
            .map_err(|_| SopError::JoinThread)?
            .map_err(SopError::WriteStdin)?;

        let output = child.wait_with_output().map_err(SopError::WaitChild)?;

        if !output.status.success() {
            let stderr = String::from_utf8_lossy(&output.stderr).to_string();
            return Err(SopError::Failed(
                self.sop.clone(),
                output.status.code().unwrap_or(999),
                stderr,
            ));
        }

        Ok(output.stdout)
    }

    fn key(&self) -> Result<Key, SopError> {
        let data =
            std::fs::read(&self.key).map_err(|err| SopError::ReadKey(self.key.clone(), err))?;
        Ok(Key::new(data))
    }
}

#[derive(Debug, thiserror::Error)]
pub enum SopError {
    #[error("failed to run {0}")]
    Invoke(PathBuf, #[source] std::io::Error),

    #[error("{0} failed with exit code {0}, stderr:\n{1}")]
    Failed(PathBuf, i32, String),

    #[error("failed to get stdin from child process handle")]
    TakeStdin,

    #[error("failed to read key from {0}")]
    ReadKey(PathBuf, #[source] std::io::Error),

    #[error("failed to join thread that writes to child stdin")]
    JoinThread,

    #[error("failed to write key to child stdin")]
    WriteStdin(#[source] std::io::Error),

    #[error("failed when waiting for child process to end")]
    WaitChild(#[source] std::io::Error),

    #[error("failed to create file {0}")]
    CreateFile(PathBuf, #[source] std::io::Error),

    #[error("failed to create a temporary directory")]
    TempDir(#[source] std::io::Error),

    #[error("failed to write to temporary file")]
    TempWrite(#[source] std::io::Error),
}
