//! The `init` sub-command.

use std::{fs::create_dir_all, path::PathBuf};

use clap::Parser;
use log::{debug, info};

use crate::{
    store::{EncryptedStore, EncryptedStoreError, Store},
    Config, LeafCommand,
};

/// Create and initialize a new password store. Use an existing key
/// for this device and extract the corresponding certificate and
/// store it in the new password store.
#[derive(Debug, Parser)]
pub struct InitCommand {
    /// Name of certificate extracted from the specified key.
    #[clap(long)]
    name: String,

    /// File containing the key to use on this device.
    #[clap(long)]
    key: PathBuf,
}

impl LeafCommand for InitCommand {
    type Error = InitError;

    fn run(&self, config: &Config) -> Result<(), InitError> {
        info!("init password store");
        debug!("{config:#?}");
        let store_dir = config.store();
        if !store_dir.exists() {
            println!("creating {}", store_dir.display());
            create_dir_all(store_dir).map_err(|err| InitError::CreateDir(store_dir.into(), err))?;
        }

        let encrypted = EncryptedStore::new(config.sop(), config.sop_decrypt(), config.store());
        encrypted.install_key(&self.key)?;
        let cert = encrypted.extract_cert()?;

        let mut store = Store::default();
        store.push_cert(&self.name, &cert);

        encrypted.write_store(&store)?;

        Ok(())
    }
}

/// Possible errors from the `init` sub-command.
#[derive(Debug, thiserror::Error)]
pub enum InitError {
    /// Can't create store directory.
    #[error("failed to create store directory {0}")]
    CreateDir(PathBuf, #[source] std::io::Error),

    /// Error from dealing with the encrypted store.
    #[error(transparent)]
    Encrypted(#[from] EncryptedStoreError),
}
