//! Use the specified SOP implementation.
//!
//! This module makes it convenient to use the SOP implementation that
//! is meant to be used by `sopass`.

use std::{
    fs::File,
    io::Write,
    path::{Path, PathBuf},
    process::{Command, Stdio},
    thread::spawn,
};

use log::info;
use tempfile::tempdir;

/// A private key in memory.
pub struct Key {
    data: Vec<u8>,
}

impl Key {
    /// Create a [`Key`] from key materiel.
    pub fn new(data: Vec<u8>) -> Self {
        Self { data }
    }

    /// The key materiel as a byte slice.
    pub fn as_bytes(&self) -> &[u8] {
        &self.data
    }
}

/// A certificate in memory.
pub struct Certificate {
    data: Vec<u8>,
}

impl Certificate {
    /// Create a [`Certificate`] from bytes.
    pub fn new(data: Vec<u8>) -> Self {
        Self { data }
    }

    /// Access a [`Certificate`] as a byte slice.
    pub fn as_bytes(&self) -> &[u8] {
        &self.data
    }
}

/// The SOP implementation and associated private key file to use.
pub struct Sop {
    sop: PathBuf,
    sop_decrypt: PathBuf,
    key: PathBuf,
}

impl Sop {
    /// Create a new [`Sop`].
    pub fn new(sop: &Path, sop_decrypt: &Path, key: &Path) -> Self {
        Self {
            sop: sop.into(),
            sop_decrypt: sop_decrypt.into(),
            key: key.into(),
        }
    }

    /// Extract a [`Certificate`] from the private key.
    pub fn extract_cert(&self) -> Result<Certificate, SopError> {
        let mut child = Command::new(&self.sop)
            .arg("extract-cert")
            .stdin(Stdio::piped())
            .stdout(Stdio::piped())
            .spawn()
            .map_err(|err| SopError::Invoke(self.sop.clone(), err))?;

        let key = self.key()?;
        let mut stdin = child.stdin.take().ok_or(SopError::TakeStdin)?;
        let writer = spawn(move || stdin.write_all(key.as_bytes()));
        writer
            .join()
            .map_err(|_| SopError::JoinThread)?
            .map_err(SopError::WriteStdin)?;

        let output = child.wait_with_output().map_err(SopError::WaitChild)?;

        if !output.status.success() {
            let stderr = String::from_utf8_lossy(&output.stderr).to_string();
            return Err(SopError::Failed(
                self.sop.clone(),
                output.status.code().unwrap_or(999),
                stderr,
            ));
        }

        Ok(Certificate::new(output.stdout))
    }

    /// Encrypt data using a set of certificates.
    pub fn encrypt(
        &self,
        data: Vec<u8>,
        certs: &[Certificate],
        output: &Path,
    ) -> Result<Vec<u8>, SopError> {
        info!("encrypt data to {}", output.display());

        let tmp = tempdir().map_err(SopError::TempDir)?;

        let mut filenames = vec![];
        for (i, cert) in certs.iter().enumerate() {
            let filename = format!("cert-{i}");
            let filename = tmp.path().join(&filename);
            std::fs::write(&filename, cert.as_bytes()).map_err(SopError::TempWrite)?;
            filenames.push(PathBuf::from(&filename));
        }

        let stdout =
            File::create(output).map_err(|err| SopError::CreateFile(output.into(), err))?;

        let mut child = Command::new(&self.sop)
            .arg("encrypt")
            .args(&filenames)
            .stdin(Stdio::piped())
            .stdout(stdout)
            .spawn()
            .map_err(|err| SopError::Invoke(self.sop.clone(), err))?;

        let mut stdin = child.stdin.take().ok_or(SopError::TakeStdin)?;
        let writer = spawn(move || stdin.write_all(&data));
        writer
            .join()
            .map_err(|_| SopError::JoinThread)?
            .map_err(SopError::WriteStdin)?;

        let output = child.wait_with_output().map_err(SopError::WaitChild)?;

        if !output.status.success() {
            let stderr = String::from_utf8_lossy(&output.stderr).to_string();
            return Err(SopError::Failed(
                self.sop.clone(),
                output.status.code().unwrap_or(999),
                stderr,
            ));
        }

        Ok(output.stdout)
    }

    /// Decrypt data using the private key specified.
    pub fn decrypt(&self, data: Vec<u8>) -> Result<Vec<u8>, SopError> {
        // For the rsoct implementation of sop_decrypt, we need to
        // give the cert file, not the key file to the `decrypt`
        // command.
        let arg = if self.sop_decrypt == Path::new("rsoct") {
            let cert = self.extract_cert()?;

            let tmp = tempdir().map_err(SopError::TempDir)?;
            let cert_filename = tmp.path().join("cert");
            std::fs::write(&cert_filename, cert.as_bytes()).unwrap();
            cert_filename
        } else {
            self.key.clone()
        };

        let mut child = Command::new(&self.sop_decrypt)
            .arg("decrypt")
            .arg(&arg)
            .stdin(Stdio::piped())
            .stdout(Stdio::piped())
            .spawn()
            .map_err(|err| SopError::Invoke(self.sop_decrypt.clone(), err))?;

        let mut stdin = child.stdin.take().ok_or(SopError::TakeStdin)?;
        let writer = spawn(move || stdin.write_all(&data));
        writer
            .join()
            .map_err(|_| SopError::JoinThread)?
            .map_err(SopError::WriteStdin)?;

        let output = child.wait_with_output().map_err(SopError::WaitChild)?;

        if !output.status.success() {
            let stderr = String::from_utf8_lossy(&output.stderr).to_string();
            return Err(SopError::Failed(
                self.sop_decrypt.clone(),
                output.status.code().unwrap_or(999),
                stderr,
            ));
        }

        Ok(output.stdout)
    }

    fn key(&self) -> Result<Key, SopError> {
        let data =
            std::fs::read(&self.key).map_err(|err| SopError::ReadKey(self.key.clone(), err))?;
        Ok(Key::new(data))
    }
}

/// Possible errors from using [`Sop`].
#[derive(Debug, thiserror::Error)]
pub enum SopError {
    /// Failed to invoke the SOP implementation, for example, if the
    /// executable doesn't exist.
    #[error("failed to run {0}")]
    Invoke(PathBuf, #[source] std::io::Error),

    /// The SOP implementation failed to do what was requested.
    #[error("{0} failed with exit code {1}, stderr:\n{2}")]
    Failed(PathBuf, i32, String),

    /// Failed to redirect stdin of SOP implementation.
    #[error("failed to get stdin from child process handle")]
    TakeStdin,

    /// Failed to read the private key file.
    #[error("failed to read key from {0}")]
    ReadKey(PathBuf, #[source] std::io::Error),

    /// Failed to join thread.
    #[error("failed to join thread that writes to child stdin")]
    JoinThread,

    /// Failed to write to SOP implementation stdin.
    #[error("failed to write key to child stdin")]
    WriteStdin(#[source] std::io::Error),

    /// Failed to wait for SOP implementation to finish.
    #[error("failed when waiting for child process to end")]
    WaitChild(#[source] std::io::Error),

    /// Failed to create file to run SOP.
    #[error("failed to create file {0}")]
    CreateFile(PathBuf, #[source] std::io::Error),

    /// Failed to create directory to run SOP.
    #[error("failed to create a temporary directory")]
    TempDir(#[source] std::io::Error),

    /// Failed to write to temporary file to run SOP.
    #[error("failed to write to temporary file")]
    TempWrite(#[source] std::io::Error),
}
