//! Parse markdown into an HTML representation.

use std::{collections::HashSet, path::Path};

use line_col::LineColLookup;
use log::trace;
use pulldown_cmark::{CodeBlockKind, Event, HeadingLevel, Options, Parser, Tag};

use crate::html::{
    as_plain_text, Attribute, BlockAttr, Content, Element, ElementTag, HtmlError, Location,
};

/// Parse Markdown text into an HTML element.
pub fn parse(filename: &Path, markdown: &str) -> Result<Element, HtmlError> {
    let mut options = Options::empty();
    options.insert(Options::ENABLE_HEADING_ATTRIBUTES);
    options.insert(Options::ENABLE_STRIKETHROUGH);
    options.insert(Options::ENABLE_TABLES);
    options.insert(Options::ENABLE_TASKLISTS);
    let p = Parser::new_ext(markdown, options).into_offset_iter();
    let linecol = LineColLookup::new(markdown);
    let mut stack = Stack::new();
    stack.push(Element::new(ElementTag::Div));
    let mut slugs = Slugs::default();
    let mut table_cell_tag = vec![];
    for (event, loc) in p {
        trace!("event {:?}", event);
        let (line, col) = linecol.get(loc.start);
        let loc = Location::new(filename, line, col);
        match event {
            Event::Start(tag) => match tag {
                Tag::Paragraph => stack.push_tag(ElementTag::P, loc),
                Tag::Heading(level, id, classes) => {
                    let tag = match level {
                        HeadingLevel::H1 => ElementTag::H1,
                        HeadingLevel::H2 => ElementTag::H2,
                        HeadingLevel::H3 => ElementTag::H3,
                        HeadingLevel::H4 => ElementTag::H4,
                        HeadingLevel::H5 => ElementTag::H5,
                        HeadingLevel::H6 => ElementTag::H6,
                    };
                    let mut h = Element::new(tag).with_location(loc);
                    if let Some(id) = id {
                        h.push_attribute(Attribute::new("id", id));
                        slugs.remember(id);
                    }
                    if !classes.is_empty() {
                        let mut names = String::new();
                        for c in classes {
                            if !names.is_empty() {
                                names.push(' ');
                            }
                            names.push_str(c);
                        }
                        h.push_attribute(Attribute::new("class", &names));
                    }
                    stack.push(h);
                }
                Tag::BlockQuote => stack.push_tag(ElementTag::Blockquote, loc),
                Tag::CodeBlock(kind) => {
                    stack.push_tag(ElementTag::Pre, loc);
                    if let CodeBlockKind::Fenced(attrs) = kind {
                        let mut e = stack.pop();
                        e.set_block_attributes(BlockAttr::parse(&attrs));
                        stack.push(e);
                    }
                }
                Tag::List(None) => stack.push_tag(ElementTag::Ul, loc),
                Tag::List(Some(start)) => {
                    let mut e = Element::new(ElementTag::Ol).with_location(loc);
                    e.push_attribute(Attribute::new("start", &format!("{}", start)));
                    stack.push(e);
                }
                Tag::Item => stack.push_tag(ElementTag::Li, loc),
                Tag::FootnoteDefinition(_) => unreachable!("{:?}", tag),
                Tag::Table(_) => {
                    stack.push_tag(ElementTag::Table, loc);
                    table_cell_tag.push(ElementTag::Td);
                }
                Tag::TableRow => stack.push_tag(ElementTag::Tr, loc),
                Tag::TableHead => {
                    stack.push_tag(ElementTag::Tr, loc);
                    table_cell_tag.push(ElementTag::Th);
                }
                Tag::TableCell => {
                    let tag = table_cell_tag.pop().unwrap();
                    table_cell_tag.push(tag);
                    stack.push_tag(tag, loc);
                }
                Tag::Emphasis => stack.push_tag(ElementTag::Em, loc),
                Tag::Strong => stack.push_tag(ElementTag::Strong, loc),
                Tag::Strikethrough => stack.push_tag(ElementTag::Del, loc),
                Tag::Link(_, url, title) => {
                    let mut link = Element::new(ElementTag::A);
                    link.push_attribute(Attribute::new("href", url.as_ref()));
                    if !title.is_empty() {
                        link.push_attribute(Attribute::new("title", title.as_ref()));
                    }
                    stack.push(link);
                }
                Tag::Image(_, url, title) => {
                    let mut e = Element::new(ElementTag::Img);
                    e.push_attribute(Attribute::new("src", url.as_ref()));
                    e.push_attribute(Attribute::new("alt", title.as_ref()));
                    if !title.is_empty() {
                        e.push_attribute(Attribute::new("title", title.as_ref()));
                    }
                    stack.push(e);
                }
            },
            Event::End(tag) => match &tag {
                Tag::Paragraph => {
                    trace!("at end of paragraph, looking for definition list use");
                    let e = stack.pop();
                    let s = as_plain_text(e.children());
                    trace!("paragraph text: {:?}", s);
                    if s.starts_with(": ") || s.contains("\n: ") {
                        return Err(HtmlError::DefinitionList(loc));
                    }
                    stack.append_child(Content::Elt(e));
                }
                Tag::Heading(_, _, _) => {
                    let mut e = stack.pop();
                    if e.attr("id").is_none() {
                        let slug = slugs.unique(&e.heading_slug());
                        let id = Attribute::new("id", &slug);
                        e.push_attribute(id);
                    }
                    stack.append_child(Content::Elt(e));
                }
                Tag::List(_)
                | Tag::Item
                | Tag::Link(_, _, _)
                | Tag::Image(_, _, _)
                | Tag::Emphasis
                | Tag::Table(_)
                | Tag::TableRow
                | Tag::TableCell
                | Tag::Strong
                | Tag::Strikethrough
                | Tag::BlockQuote
                | Tag::CodeBlock(_) => {
                    let e = stack.pop();
                    stack.append_child(Content::Elt(e));
                }
                Tag::TableHead => {
                    let e = stack.pop();
                    stack.append_child(Content::Elt(e));
                    assert!(!table_cell_tag.is_empty());
                    table_cell_tag.pop();
                }
                Tag::FootnoteDefinition(_) => unreachable!("{:?}", tag),
            },
            Event::Text(s) => stack.append_str(s.as_ref()),
            Event::Code(s) => {
                let mut code = Element::new(ElementTag::Code);
                code.push_child(Content::Text(s.to_string()));
                stack.append_element(code);
            }
            Event::Html(s) => stack.append_child(Content::Html(s.to_string())),
            Event::FootnoteReference(s) => trace!("footnote ref {:?}", s),
            Event::SoftBreak => stack.append_str("\n"),
            Event::HardBreak => stack.append_element(Element::new(ElementTag::Br)),
            Event::Rule => stack.append_element(Element::new(ElementTag::Hr)),
            Event::TaskListMarker(done) => {
                let marker = if done {
                    "\u{2612} " // Unicode for box with X
                } else {
                    "\u{2610} " // Unicode for empty box
                };
                stack.append_str(marker);
            }
        }
    }

    let mut body = stack.pop();
    assert!(stack.is_empty());
    body.fix_up_img_alt();
    Ok(body)
}

struct Stack {
    stack: Vec<Element>,
}

impl Stack {
    fn new() -> Self {
        Self { stack: vec![] }
    }

    fn is_empty(&self) -> bool {
        self.stack.is_empty()
    }

    fn push(&mut self, e: Element) {
        trace!("pushed {:?}", e);
        self.stack.push(e);
    }

    fn push_tag(&mut self, tag: ElementTag, loc: Location) {
        self.push(Element::new(tag).with_location(loc));
    }

    fn pop(&mut self) -> Element {
        let e = self.stack.pop().unwrap();
        trace!("popped {:?}", e);
        e
    }

    fn append_child(&mut self, child: Content) {
        trace!("appended {:?}", child);
        let mut parent = self.stack.pop().unwrap();
        parent.push_child(child);
        self.stack.push(parent);
    }

    fn append_str(&mut self, text: &str) {
        self.append_child(Content::Text(text.into()));
    }

    fn append_element(&mut self, e: Element) {
        self.append_child(Content::Elt(e));
    }
}

#[derive(Debug, Default)]
struct Slugs {
    slugs: HashSet<String>,
}

impl Slugs {
    const MAX: usize = 8;

    fn remember(&mut self, slug: &str) {
        self.slugs.insert(slug.into());
    }

    fn unique(&mut self, candidate: &str) -> String {
        let slug = self.helper(candidate);
        self.remember(&slug);
        slug
    }

    fn helper(&mut self, candidate: &str) -> String {
        let mut slug0 = String::new();
        for c in candidate.chars() {
            if slug0.len() >= Self::MAX {
                break;
            }
            slug0.push(c);
        }

        if !self.slugs.contains(&slug0) {
            return slug0.to_string();
        }

        let mut i = 0;
        loop {
            i += 1;
            let slug = format!("{}{}", slug0, i);
            if !self.slugs.contains(&slug) {
                return slug;
            }
        }
    }
}
